<?php

defined( 'ABSPATH' ) || exit;

/**
 * Custom Email Reminder Template Data Store CPT
 * 
 * @class ASP_ERWS_Custom_Template_Data_Store_CPT
 * @package Class
 */
class ASP_ERWS_Custom_Template_Data_Store_CPT extends WC_Data_Store_WP implements WC_Object_Data_Store_Interface {

	/**
	 * Internal meta type used to store custom email reminder template data.
	 *
	 * @var string
	 */
	protected $meta_type = 'post';

	/**
	 * Data stored in meta keys, but not considered "meta" for an custom email reminder template.
	 *
	 * @var array
	 */
	protected $internal_meta_keys = array();

	/**
	 * Data stored in meta key to props.
	 *
	 * @var array
	 */
	protected $internal_meta_key_to_props = array(
		'_version'                 => 'version',
		'_slug'                    => 'slug',
		'_email_slug'              => 'email_slug',
		'_email_sending_interval'  => 'email_sending_interval',
		'_email_subject'           => 'email_subject',
		'_email_heading'           => 'email_heading',
		'_criteria_product_filter' => 'criteria_product_filter',
		'_criteria_user_filter'    => 'criteria_user_filter',
		'_criteria_products'       => 'criteria_products',
		'_criteria_product_cats'   => 'criteria_product_cats',
		'_criteria_users'          => 'criteria_users',
		'_criteria_user_roles'     => 'criteria_user_roles',
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->internal_meta_keys = array_merge( $this->internal_meta_keys, array_keys( $this->internal_meta_key_to_props ) );
	}

	/*
	  |--------------------------------------------------------------------------
	  | CRUD Methods
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Method to create a new ID in the database from the new changes.
	 * 
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 */
	public function create( &$custom_template ) {
		$custom_template->set_version( ASP_ERWS_VERSION );
		$custom_template->set_date_created( time() );

		$id = wp_insert_post( array(
			'post_date'     => gmdate( 'Y-m-d H:i:s', $custom_template->get_date_created( 'edit' )->getOffsetTimestamp() ),
			'post_date_gmt' => gmdate( 'Y-m-d H:i:s', $custom_template->get_date_created( 'edit' )->getTimestamp() ),
			'post_type'     => $custom_template->get_type(),
			'post_status'   => $this->get_post_status( $custom_template ),
			'ping_status'   => 'closed',
			'post_author'   => 1,
			'post_title'    => $this->get_post_title( $custom_template ),
			'post_content'  => $custom_template->get_email_content(),
			'post_parent'   => 0,
				), true );

		if ( $id && ! is_wp_error( $id ) ) {
			$custom_template->set_id( $id );
			$this->update_post_meta( $custom_template );
			$custom_template->save_meta_data();
			$custom_template->apply_changes();
			$this->clear_caches( $custom_template );

			/**
			 * New custom email reminder template is created.
			 * 
			 * @since 1.8.0
			 */
			do_action( 'asp_erws_new_custom_template', $custom_template->get_id(), $custom_template );
		}
	}

	/**
	 * Method to read data from the database.
	 * 
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 */
	public function read( &$custom_template ) {
		$custom_template->set_defaults();
		$post = get_post( $custom_template->get_id() );

		if ( ! $custom_template->get_id() || ! $post || $post->post_type !== $custom_template->get_type() ) {
			throw new Exception( esc_html__( 'Invalid custom email reminder template.', 'email-reminders-for-woocommerce-subscriptions' ) );
		}

		$custom_template->set_props(
				array(
					'name'          => $post->post_title,
					'date_created'  => $this->string_to_timestamp( $post->post_date_gmt ),
					'date_modified' => $this->string_to_timestamp( $post->post_modified_gmt ),
					'status'        => $post->post_status,
					'email_content' => $post->post_content,
				)
		);

		$this->read_custom_template_data( $custom_template, $post );
		$custom_template->read_meta_data();
		$custom_template->set_object_read( true );
	}

	/**
	 * Method to update changes in the database.
	 * 
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 */
	public function update( &$custom_template ) {
		$custom_template->save_meta_data();
		$custom_template->set_version( ASP_ERWS_VERSION );

		if ( ! $custom_template->get_date_created( 'edit' ) ) {
			$custom_template->set_date_created( time() );
		}

		$changes = $custom_template->get_changes();

		// Only update the post when the post data changes.
		if ( array_intersect( array( 'date_created', 'date_modified', 'status', 'name', 'email_content' ), array_keys( $changes ) ) ) {
			$post_data = array(
				'post_date'         => gmdate( 'Y-m-d H:i:s', $custom_template->get_date_created( 'edit' )->getOffsetTimestamp() ),
				'post_date_gmt'     => gmdate( 'Y-m-d H:i:s', $custom_template->get_date_created( 'edit' )->getTimestamp() ),
				'post_status'       => $this->get_post_status( $custom_template ),
				'post_title'        => $this->get_post_title( $custom_template ),
				'post_content'      => $custom_template->get_email_content(),
				'post_modified'     => isset( $changes[ 'date_modified' ] ) ? gmdate( 'Y-m-d H:i:s', $custom_template->get_date_modified( 'edit' )->getOffsetTimestamp() ) : current_time( 'mysql' ),
				'post_modified_gmt' => isset( $changes[ 'date_modified' ] ) ? gmdate( 'Y-m-d H:i:s', $custom_template->get_date_modified( 'edit' )->getTimestamp() ) : current_time( 'mysql', 1 ),
			);

			/**
			 * When updating this object, to prevent infinite loops, use $wpdb
			 * to update data, since wp_update_post spawns more calls to the
			 * save_post action.
			 *
			 * This ensures hooks are fired by either WP itself (admin screen save),
			 * or an update purely from CRUD.
			 */
			if ( doing_action( 'save_post' ) ) {
				$GLOBALS[ 'wpdb' ]->update( $GLOBALS[ 'wpdb' ]->posts, $post_data, array( 'ID' => $custom_template->get_id() ) );
				clean_post_cache( $custom_template->get_id() );
			} else {
				wp_update_post( array_merge( array( 'ID' => $custom_template->get_id() ), $post_data ) );
			}

			$custom_template->read_meta_data( true ); // Refresh internal meta data, in case things were hooked into `save_post` or another WP hook.
		}

		$this->update_post_meta( $custom_template );
		$custom_template->apply_changes();
		$this->clear_caches( $custom_template );
	}

	/**
	 * Delete an object, set the ID to 0.
	 *
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 * @param array $args Array of args to pass to the delete method.
	 */
	public function delete( &$custom_template, $args = array() ) {
		$id   = $custom_template->get_id();
		$args = wp_parse_args( $args, array(
			'force_delete' => false,
				) );

		if ( ! $id ) {
			return;
		}

		if ( $args[ 'force_delete' ] ) {
			wp_delete_post( $id );
			$custom_template->set_id( 0 );

			/**
			 * Custom email reminder template is deleted.
			 * 
			 * @since 1.8.0
			 */
			do_action( 'asp_erws_delete_custom_template', $id );
		} else {
			wp_trash_post( $id );
			$custom_template->set_status( 'trash' );

			/**
			 * Custom email reminder template is trashed.
			 * 
			 * @since 1.8.0
			 */
			do_action( 'asp_erws_trash_custom_template', $id );
		}
	}

	/*
	  |--------------------------------------------------------------------------
	  | Additional Methods
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Get the status to save to the post object.
	 *
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 * @return string
	 */
	protected function get_post_status( $custom_template ) {
		$post_status = $custom_template->get_status( 'edit' );

		if ( ! $post_status ) {
			/**
			 * Get the custom email reminder template default status.
			 * 
			 * @since 1.8.0
			 */
			$post_status = apply_filters( 'asp_erws_default_custom_template_status', 'active' );
		}

		if ( in_array( 'asp-' . $post_status, $custom_template->get_valid_statuses() ) ) {
			$post_status = 'asp-' . $post_status;
		}

		return $post_status;
	}

	/**
	 * Get the title to save to the post object.
	 *
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 * @return string
	 */
	protected function get_post_title( $custom_template ) {
		return $custom_template->get_name() ? $custom_template->get_name() : __( 'Custom Subscription Email Reminder Template', 'email-reminders-for-woocommerce-subscriptions' );
	}

	/**
	 * Read custom email reminder template data from the database.
	 *
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 * @param object $post_object Post object.
	 */
	protected function read_custom_template_data( &$custom_template, $post_object ) {
		foreach ( $this->internal_meta_key_to_props as $meta_key => $prop ) {
			$setter = "set_$prop";
			if ( is_callable( array( $custom_template, $setter ) ) && metadata_exists( 'post', $custom_template->get_id(), "$meta_key" ) ) {
				$custom_template->{$setter}( get_post_meta( $custom_template->get_id(), "$meta_key", true ) );
			}
		}
	}

	/**
	 * Update meta data in, or delete it from, the database.
	 * As WC defined this method @since 3.6.0 and so we reuse this method here to make it compatible with v3.5.x too.
	 *
	 * @param WC_Data $object The WP_Data object
	 * @param string  $meta_key Meta key to update.
	 * @param mixed   $meta_value Value to save.
	 * @return bool True if updated/deleted.
	 */
	protected function update_or_delete_post_meta( $object, $meta_key, $meta_value ) {
		if ( is_callable( array( get_parent_class( $this ), 'update_or_delete_post_meta' ) ) ) {
			$updated = parent::update_or_delete_post_meta( $object, $meta_key, $meta_value );
		} elseif ( in_array( $meta_value, array( array(), '' ), true ) ) {
			$updated = delete_post_meta( $object->get_id(), $meta_key );
		} else {
			$updated = update_post_meta( $object->get_id(), $meta_key, $meta_value );
		}

		return ( bool ) $updated;
	}

	/**
	 * Helper method that updates all the post meta for an custom email reminder template.
	 * 
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 */
	protected function update_post_meta( &$custom_template ) {
		$updated_props   = array();
		$props_to_update = $this->get_props_to_update( $custom_template, $this->internal_meta_key_to_props );

		foreach ( $props_to_update as $meta_key => $prop ) {
			$getter = "get_$prop";
			$value  = is_callable( array( $custom_template, $getter ) ) ? $custom_template->{$getter}( 'edit' ) : '';
			$value  = is_string( $value ) ? wp_slash( $value ) : $value;

			$updated = $this->update_or_delete_post_meta( $custom_template, $meta_key, $value );
			if ( $updated ) {
				$updated_props[] = $prop;
			}
		}

		/**
		 * Custom email reminder template object updated its props.
		 * 
		 * @since 1.8.0
		 */
		do_action( 'asp_erws_custom_template_object_updated_props', $custom_template, $updated_props );
	}

	/**
	 * Clear any caches.
	 *
	 * @param  ASP_ERWS_Custom_Template $custom_template
	 */
	protected function clear_caches( &$custom_template ) {
		clean_post_cache( $custom_template->get_id() );
	}

	/**
	 * Converts a WP post date string into a timestamp.
	 *
	 * @param  string $time_string The WP post date string.
	 * @return int|null The date string converted to a timestamp or null.
	 */
	protected function string_to_timestamp( $time_string ) {
		return '0000-00-00 00:00:00' !== $time_string ? wc_string_to_timestamp( $time_string ) : null;
	}
}
