<?php

defined( 'ABSPATH' ) || exit;

/**
 * Custom Email Reminder Template.
 * 
 * @class ASP_ERWS_Custom_Template
 * @package Class
 */
class ASP_ERWS_Custom_Template extends WC_Data {

	/**
	 * Custom email reminder template Data array. 
	 *
	 * @var array
	 */
	protected $data = array(
		'name'                    => '',
		'status'                  => '',
		'version'                 => '',
		'slug'                    => '',
		'email_slug'              => '',
		'email_sending_interval'  => '',
		'email_subject'           => '',
		'email_heading'           => '',
		'email_content'           => '',
		'date_created'            => null,
		'date_modified'           => null,
		'priority'                => 0,
		'criteria_user_filter'    => 'all-users',
		'criteria_users'          => array(),
		'criteria_user_roles'     => array(),
		'criteria_product_filter' => 'all-products',
		'criteria_products'       => array(),
		'criteria_product_cats'   => array(),
	);

	/**
	 * Stores meta in cache for future reads.
	 *
	 * A group must be set to to enable caching.
	 *
	 * @var string
	 */
	protected $cache_group = 'asp_erws_custom_templates';

	/**
	 * Which data store to load.
	 *
	 * @var string
	 */
	protected $data_store_name = 'asp_erws_custom_template';

	/**
	 * This is the name of this object type.
	 *
	 * @var string
	 */
	protected $object_type = 'asp_erws_custom_tmpl';

	/**
	 * Get the custom email reminder template if ID is passed, otherwise the custom email reminder template is new and empty.
	 *
	 * @param  int|object|ASP_ERWS_Custom_Template $custom_template Custom email reminder template to read.
	 */
	public function __construct( $custom_template = 0 ) {
		parent::__construct( $custom_template );

		if ( is_numeric( $custom_template ) && $custom_template > 0 ) {
			$this->set_id( $custom_template );
		} elseif ( $custom_template instanceof self ) {
			$this->set_id( $custom_template->get_id() );
		} elseif ( ! empty( $custom_template->ID ) ) {
			$this->set_id( $custom_template->ID );
		} else {
			$this->set_object_read( true );
		}

		$this->data_store = WC_Data_Store::load( $this->data_store_name );

		if ( $this->get_id() > 0 ) {
			$this->data_store->read( $this );
		}
	}

	/**
	 * Get internal type.
	 *
	 * @return string
	 */
	public function get_type() {
		return 'asp_erws_custom_tmpl';
	}

	/**
	 * Get all valid statuses for this custom email reminder template.
	 *
	 * @return array Internal status keys e.g. 'asp-active'
	 */
	public function get_valid_statuses() {
		return array_keys( asp_erws_get_custom_template_statuses() );
	}

	/**
	 * Updates status of custom email reminder template immediately.
	 *
	 * @uses ASP_ERWS_Custom_Template::set_status()
	 * @param string $new_status    Status to change the custom email reminder template to. No internal asp- prefix is required.
	 * @return bool
	 */
	public function update_status( $new_status ) {
		if ( ! $this->get_id() ) { // Custom email reminder template must exist.
			return false;
		}

		try {
			$this->set_status( $new_status );
			$this->save();
		} catch ( Exception $e ) {
			$this->handle_exception( $e, sprintf( 'Error updating status for custom email reminder template #%d', $this->get_id() ) );
			return false;
		}
		return true;
	}

	/**
	 * Log an error about this custom email reminder template is exception is encountered.
	 *
	 * @param Exception $e Exception object.
	 * @param string    $message Message regarding exception thrown.
	 */
	protected function handle_exception( $e, $message = 'Error' ) {
		wc_get_logger()->error(
				$message,
				array(
					'custom-template' => $this,
					'error'           => $e,
				)
		);
	}

	/*
	  |--------------------------------------------------------------------------
	  | Conditionals
	  |--------------------------------------------------------------------------
	  |
	  | Checks if a condition is true or false.
	  |
	 */

	/**
	 * Checks the custom email reminder template status against a passed in status.
	 *
	 * @param array|string $status Status to check.
	 * @return bool
	 */
	public function has_status( $status ) {
		/**
		 * Checks if the custom email reminder template has status.
		 * 
		 * @since 1.8.0
		 */
		return apply_filters( 'asp_erws_custom_template_has_status', ( is_array( $status ) && in_array( $this->get_status(), $status, true ) ) || $this->get_status() === $status, $this, $status );
	}

	/*
	  |--------------------------------------------------------------------------
	  | URLs and Endpoints
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Get's the URL to edit the custom email reminder template in the backend.
	 *
	 * @return string
	 */
	public function get_edit_custom_template_url() {
		/**
		 * Get the admin custom email reminder template edit URL.
		 * 
		 * @since 1.8.0
		 */
		return apply_filters( 'asp_erws_get_admin_edit_custom_template_url', get_admin_url( null, 'post.php?post=' . $this->get_id() . '&action=edit' ), $this );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Getters
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Get name.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_name( $context = 'view' ) {
		return $this->get_prop( 'name', $context );
	}

	/**
	 * Get version.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_version( $context = 'view' ) {
		return $this->get_prop( 'version', $context );
	}

	/**
	 * Get slug.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_slug( $context = 'view' ) {
		return $this->get_prop( 'slug', $context );
	}

	/**
	 * Get email slug.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_email_slug( $context = 'view' ) {
		return $this->get_prop( 'email_slug', $context );
	}

	/**
	 * Get email sending interval.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_email_sending_interval( $context = 'view' ) {
		return $this->get_prop( 'email_sending_interval', $context );
	}

	/**
	 * Get email subject.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_email_subject( $context = 'view' ) {
		return $this->get_prop( 'email_subject', $context );
	}

	/**
	 * Get email heading.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_email_heading( $context = 'view' ) {
		return $this->get_prop( 'email_heading', $context );
	}

	/**
	 * Get email content.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_email_content( $context = 'view' ) {
		return $this->get_prop( 'email_content', $context );
	}

	/**
	 * Get date created.
	 *
	 * @param  string $context View or edit context.
	 * @return WC_DateTime|NULL object if the date is set or null if there is no date.
	 */
	public function get_date_created( $context = 'view' ) {
		return $this->get_prop( 'date_created', $context );
	}

	/**
	 * Get date modified.
	 *
	 * @param  string $context View or edit context.
	 * @return WC_DateTime|NULL object if the date is set or null if there is no date.
	 */
	public function get_date_modified( $context = 'view' ) {
		return $this->get_prop( 'date_modified', $context );
	}

	/**
	 * Get priority.
	 *
	 * @param  string $context View or edit context.
	 * @return int
	 */
	public function get_priority( $context = 'view' ) {
		return $this->get_prop( 'priority', $context );
	}

	/**
	 * Return the custom email reminder template statuses without asp- internal prefix.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_status( $context = 'view' ) {
		$status = $this->get_prop( 'status', $context );

		if ( empty( $status ) && 'view' === $context ) {
			/**
			 * In view context, return the default status if no status has been set.
			 * 
			 * @since 1.8.0
			 */
			$status = apply_filters( 'asp_erws_default_custom_template_status', 'active' );
		}
		return $status;
	}

	/**
	 * Get custom email reminder template criteria user filter.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_criteria_user_filter( $context = 'view' ) {
		return $this->get_prop( 'criteria_user_filter', $context );
	}

	/**
	 * Get custom email reminder template criteria users.
	 *
	 * @param  string $context View or edit context.
	 * @return array
	 */
	public function get_criteria_users( $context = 'view' ) {
		return $this->get_prop( 'criteria_users', $context );
	}

	/**
	 * Get custom email reminder template criteria user roles.
	 *
	 * @param  string $context View or edit context.
	 * @return array
	 */
	public function get_criteria_user_roles( $context = 'view' ) {
		return $this->get_prop( 'criteria_user_roles', $context );
	}

	/**
	 * Get custom email reminder template criteria product filter.
	 *
	 * @param  string $context View or edit context.
	 * @return string
	 */
	public function get_criteria_product_filter( $context = 'view' ) {
		return $this->get_prop( 'criteria_product_filter', $context );
	}

	/**
	 * Get custom email reminder template criteria products.
	 *
	 * @param  string $context View or edit context.
	 * @return array
	 */
	public function get_criteria_products( $context = 'view' ) {
		return $this->get_prop( 'criteria_products', $context );
	}

	/**
	 * Get custom email reminder template criteria product cats.
	 *
	 * @param  string $context View or edit context.
	 * @return array
	 */
	public function get_criteria_product_cats( $context = 'view' ) {
		return $this->get_prop( 'criteria_product_cats', $context );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Setters
	  |--------------------------------------------------------------------------
	  |
	  | Functions for setting custom email reminder template data. These should not update anything in the
	  | database itself and should only change what is stored in the class
	  | object.
	 */

	/**
	 * Set name.
	 *
	 * @param string $value Value to set.
	 */
	public function set_name( $value ) {
		$this->set_prop( 'name', $value );
	}

	/**
	 * Set status.
	 *
	 * @param string $new_status Status to change the custom email reminder template to. No internal asp- prefix is required.
	 * @return array details of change
	 */
	public function set_status( $new_status ) {
		$old_status        = $this->get_status();
		$new_status        = asp_erws_trim_post_status( $new_status );
		$status_exceptions = array( 'auto-draft', 'trash' );

		// If setting the status, ensure it's set to a valid status.
		if ( true === $this->object_read ) {
			// Only allow valid new status.
			if ( ! in_array( 'asp-' . $new_status, $this->get_valid_statuses(), true ) && ! in_array( $new_status, $status_exceptions, true ) ) {
				$new_status = 'active';
			}

			// If the old status is set but unknown (e.g. draft) assume its active for action usage.
			if ( $old_status && ! in_array( 'asp-' . $old_status, $this->get_valid_statuses(), true ) && ! in_array( $old_status, $status_exceptions, true ) ) {
				$old_status = 'active';
			}
		}

		$this->set_prop( 'status', $new_status );

		return array(
			'from' => $old_status,
			'to'   => $new_status,
		);
	}

	/**
	 * Set version.
	 *
	 * @param string $value Value to set.
	 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
	 */
	public function set_version( $value ) {
		$this->set_prop( 'version', $value );
	}

	/**
	 * Set slug.
	 *
	 * @param string $value Value to set.
	 */
	public function set_slug( $value ) {
		$this->set_prop( 'slug', $value );
	}

	/**
	 * Set email slug.
	 *
	 * @param string $value Value to set.
	 */
	public function set_email_slug( $value ) {
		$this->set_prop( 'email_slug', $value );
	}

	/**
	 * Set email sending interval.
	 *
	 * @param string $value Value to set.
	 */
	public function set_email_sending_interval( $value ) {
		$this->set_prop( 'email_sending_interval', $value );
	}

	/**
	 * Set email subject.
	 *
	 * @param string $value Value to set.
	 */
	public function set_email_subject( $value ) {
		$this->set_prop( 'email_subject', $value );
	}

	/**
	 * Set email heading.
	 *
	 * @param string $value Value to set.
	 */
	public function set_email_heading( $value ) {
		$this->set_prop( 'email_heading', $value );
	}

	/**
	 * Set email content.
	 *
	 * @param string $value Value to set.
	 */
	public function set_email_content( $value ) {
		$this->set_prop( 'email_content', $value );
	}

	/**
	 * Set date created.
	 *
	 * @param  string|integer|null $date UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if there is no date.
	 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
	 */
	public function set_date_created( $date = null ) {
		$this->set_date_prop( 'date_created', $date );
	}

	/**
	 * Set date modified.
	 *
	 * @param  string|integer|null $date UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if there is no date.
	 * @throws WC_Data_Exception Exception may be thrown if value is invalid.
	 */
	public function set_date_modified( $date = null ) {
		$this->set_date_prop( 'date_modified', $date );
	}

	/**
	 * Set priority.
	 *
	 * @param int $value Value to set.
	 */
	public function set_priority( $value ) {
		$this->set_prop( 'priority', absint( $value ) );
	}

	/**
	 * Set custom email reminder template criteria user filter.
	 *
	 * @param string $value
	 */
	public function set_criteria_user_filter( $value ) {
		$this->set_prop( 'criteria_user_filter', $value );
	}

	/**
	 * Set custom email reminder template criteria users.
	 *
	 * @param array $value
	 */
	public function set_criteria_users( $value ) {
		$value = is_array( $value ) ? $value : array();
		$this->set_prop( 'criteria_users', $value );
	}

	/**
	 * Set custom email reminder template criteria user roles.
	 *
	 * @param array $value
	 */
	public function set_criteria_user_roles( $value ) {
		$value = is_array( $value ) ? $value : array();
		$this->set_prop( 'criteria_user_roles', $value );
	}

	/**
	 * Set custom email reminder template criteria product filter.
	 *
	 * @param string $value
	 */
	public function set_criteria_product_filter( $value ) {
		$this->set_prop( 'criteria_product_filter', $value );
	}

	/**
	 * Set custom email reminder template criteria products.
	 *
	 * @param array $value
	 */
	public function set_criteria_products( $value ) {
		$value = is_array( $value ) ? $value : array();
		$this->set_prop( 'criteria_products', $value );
	}

	/**
	 * Set custom email reminder template criteria products cats.
	 *
	 * @param array $value
	 */
	public function set_criteria_product_cats( $value ) {
		$value = is_array( $value ) ? $value : array();
		$this->set_prop( 'criteria_product_cats', $value );
	}
}
