<?php

defined( 'ABSPATH' ) || exit;

/**
 * Admin Post Types handler.
 * 
 * @class ASP_ERWS_Admin_Post_Types
 * @package Class
 */
class ASP_ERWS_Admin_Post_Types {

	/**
	 * Is meta boxes saved once?
	 *
	 * @var bool
	 */
	private static $saved_meta_boxes = false;

	/**
	 * Init ASP_ERWS_Admin_Post_Types.
	 */
	public static function init() {
		self::load_list_types();

		add_filter( 'wp_untrash_post_status', __CLASS__ . '::wp_untrash_post_status', 10, 3 );
		add_filter( 'post_updated_messages', __CLASS__ . '::post_updated_messages' );
		add_filter( 'bulk_post_updated_messages', __CLASS__ . '::bulk_post_updated_messages', 10, 2 );
		add_filter( 'enter_title_here', __CLASS__ . '::enter_title_here', 1, 2 );
		add_action( 'add_meta_boxes', __CLASS__ . '::add_meta_boxes', 100 );
		add_action( 'add_meta_boxes', __CLASS__ . '::remove_meta_boxes', 50 );
		add_action( 'save_post', __CLASS__ . '::save_meta_boxes', 1, 2 );
		add_action( 'asp_erws_process_asp_erws_custom_tmpl_posted_meta', 'ASP_ERWS_Meta_Box_Custom_Template_Actions::save', 10, 3 );
	}

	/**
	 * Load list types.
	 */
	protected static function load_list_types() {
		if ( ! class_exists( 'WC_Admin_List_Table', false ) ) {
			include_once WC_ABSPATH . 'includes/admin/list-tables/abstract-class-wc-admin-list-table.php';
		}

		include_once 'list-tables/class-asp-erws-admin-list-table-custom-templates.php';
	}

	/**
	 * Ensure statuses are correctly reassigned when restoring our posts.
	 *
	 * @param string $new_status      The new status of the post being restored.
	 * @param int    $post_id         The ID of the post being restored.
	 * @param string $previous_status The status of the post at the point where it was trashed.
	 * @return string
	 */
	public static function wp_untrash_post_status( $new_status, $post_id, $previous_status ) {
		if ( in_array( get_post_type( $post_id ), array( 'asp_erws_custom_tmpl' ), true ) ) {
			$new_status = $previous_status;
		}

		return $new_status;
	}

	/**
	 * Change messages when a post type is updated.
	 *
	 * @param  array $messages Array of messages.
	 * @return array
	 */
	public static function post_updated_messages( $messages ) {
		$messages[ 'asp_erws_custom_tmpl' ] = array(
			0 => '', // Unused. Messages start at index 1.
			1 => __( 'Custom email reminder template updated.', 'email-reminders-for-woocommerce-subscriptions' ),
			4 => __( 'Custom email reminder template updated.', 'email-reminders-for-woocommerce-subscriptions' ),
			6 => __( 'Custom email reminder template updated.', 'email-reminders-for-woocommerce-subscriptions' ),
			7 => __( 'Custom email reminder template saved.', 'email-reminders-for-woocommerce-subscriptions' ),
			8 => __( 'Custom email reminder template submitted.', 'email-reminders-for-woocommerce-subscriptions' ),
		);

		return $messages;
	}

	/**
	 * Specify custom bulk actions messages for different post types.
	 *
	 * @param  array $bulk_messages Array of messages.
	 * @param  array $bulk_counts Array of how many objects were updated.
	 * @return array
	 */
	public static function bulk_post_updated_messages( $bulk_messages, $bulk_counts ) {
		$bulk_messages[ 'asp_erws_custom_tmpl' ] = array(
			/* translators: %s: custom email reminder template count */
			'updated'   => _n( '%s custom email reminder template updated.', '%s custom email reminder templates updated.', $bulk_counts[ 'updated' ], 'email-reminders-for-woocommerce-subscriptions' ),
			/* translators: %s: custom email reminder template count */
			'locked'    => _n( '%s custom email reminder template not updated, somebody is editing it.', '%s custom email reminder templates not updated, somebody is editing them.', $bulk_counts[ 'locked' ], 'email-reminders-for-woocommerce-subscriptions' ),
			/* translators: %s: custom email reminder template count */
			'deleted'   => _n( '%s custom email reminder template permanently deleted.', '%s custom email reminder templates permanently deleted.', $bulk_counts[ 'deleted' ], 'email-reminders-for-woocommerce-subscriptions' ),
			/* translators: %s: custom email reminder template count */
			'trashed'   => _n( '%s custom email reminder template moved to the Trash.', '%s custom email reminder templates moved to the Trash.', $bulk_counts[ 'trashed' ], 'email-reminders-for-woocommerce-subscriptions' ),
			/* translators: %s: custom email reminder template count */
			'untrashed' => _n( '%s custom email reminder template restored from the Trash.', '%s custom email reminder templates restored from the Trash.', $bulk_counts[ 'untrashed' ], 'email-reminders-for-woocommerce-subscriptions' ),
		);

		return $bulk_messages;
	}

	/**
	 * Change title boxes in admin.
	 *
	 * @param string  $text Text to shown.
	 * @param WP_Post $post Current post object.
	 * @return string
	 */
	public static function enter_title_here( $text, $post ) {
		switch ( $post->post_type ) {
			case 'asp_erws_custom_tmpl':
				$text = esc_html__( 'Template name', 'email-reminders-for-woocommerce-subscriptions' );
				break;
		}

		return $text;
	}

	/**
	 * Add Metaboxes.
	 */
	public static function add_meta_boxes() {
		add_meta_box( 'asp_erws_custom_template_data', __( 'Template data', 'email-reminders-for-woocommerce-subscriptions' ), 'ASP_ERWS_Meta_Box_Custom_Template_Data::output', 'asp_erws_custom_tmpl', 'normal', 'high' );
		add_meta_box( 'asp_erws_custom_template_actions', __( 'Actions', 'email-reminders-for-woocommerce-subscriptions' ), 'ASP_ERWS_Meta_Box_Custom_Template_Actions::output', 'asp_erws_custom_tmpl', 'side', 'high' );
		add_meta_box( 'asp_erws_custom_template_test_email', __( 'Test email', 'email-reminders-for-woocommerce-subscriptions' ), 'ASP_ERWS_Meta_Box_Custom_Template_Test_Email::output', 'asp_erws_custom_tmpl', 'side', 'low' );
		add_meta_box( 'asp_erws_custom_template_shortcodes', __( 'Shortcodes supported', 'email-reminders-for-woocommerce-subscriptions' ), 'ASP_ERWS_Meta_Box_Custom_Template_Shortcodes::output', 'asp_erws_custom_tmpl', 'normal', 'low' );
	}

	/**
	 * Remove Metaboxes.
	 */
	public static function remove_meta_boxes() {
		remove_meta_box( 'commentsdiv', 'asp_erws_custom_tmpl', 'normal' );
		remove_meta_box( 'submitdiv', 'asp_erws_custom_tmpl', 'side' );
	}

	/**
	 * Check if we're saving, the trigger an action based on the post type.
	 *
	 * @param  int    $post_id Post ID.
	 * @param  object $post Post object.
	 */
	public static function save_meta_boxes( $post_id, $post ) {
		$post_id = absint( $post_id );

		// $post_id and $post are required
		if ( empty( $post_id ) || empty( $post ) || self::$saved_meta_boxes ) {
			return;
		}

		// Dont' save meta boxes for revisions or autosaves.
		if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || is_int( wp_is_post_revision( $post ) ) || is_int( wp_is_post_autosave( $post ) ) ) {
			return;
		}

		// Check the nonce.
		if ( empty( $_POST[ 'asp_erws_save_meta_nonce' ] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST[ 'asp_erws_save_meta_nonce' ] ) ), 'asp_erws_save_data' ) ) {
			return;
		}

		// Check the post being saved == the $post_id to prevent triggering this call for other save_post events.
		$posted = $_POST;
		if ( empty( $posted[ 'post_ID' ] ) || absint( $posted[ 'post_ID' ] ) !== $post_id ) {
			return;
		}

		// Check user has permission to edit.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		self::$saved_meta_boxes = true;

		if ( in_array( $post->post_type, array( 'asp_erws_custom_tmpl' ), true ) ) {
			/**
			 * Process our post types save.
			 * 
			 * @since 1.8.0
			 */
			do_action( "asp_erws_process_{$post->post_type}_posted_meta", $post_id, $post, $posted );
		}
	}
}
