<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Subscription Scheduled Activation Reminder Email.
 *
 * @class ASP_ERWS_Email_Customer_Subscription_CSD_Scheduled_Activation_Reminder
 * @extends WC_Email
 */
class ASP_ERWS_Email_Customer_Subscription_CSD_Scheduled_Activation_Reminder extends WC_Email {

	/**
	 * Gets reminding day.
	 * 
	 * @var int
	 */
	public $remind_day;

	/**
	 * Custom template ID.
	 *
	 * @var int
	 */
	public $custom_template_id;

	/**
	 * Strings to find/replace in custom template supported email.
	 *
	 * @var array Supports
	 */
	public $custom_template_shortcodes = array(
		'{customer_name}'         => '',
		'{customer_first_name}'   => '',
		'{customer_last_name}'    => '',
		'{customer_details}'      => '',
		'{view_subscription_url}' => '',
		'{subscription_details}'  => '',
		'{start_date}'            => '',
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id             = 'asp_erws_customer_subscription_csd_scheduled_activation_reminder';
		$this->title          = __( 'Subscription Scheduled Activation Reminder', 'email-reminders-for-woocommerce-subscriptions' );
		$this->description    = __( 'This reminder is sent to the customer before the scheduled subscription becomes active.', 'email-reminders-for-woocommerce-subscriptions' );
		$this->customer_email = true;

		$this->heading = __( 'Your subscription is about to become active', 'email-reminders-for-woocommerce-subscriptions' );
		$this->subject = __( '[{blogname}] Your subscription is about to become active', 'email-reminders-for-woocommerce-subscriptions' );

		$this->template_html  = 'emails/csd-customer-subscription-scheduled-activation-reminder.php';
		$this->template_plain = 'emails/plain/csd-customer-subscription-scheduled-activation-reminder.php';
		$this->template_base  = ASP_ERWS()->plugin_path() . '/templates/';

		add_action( 'asp_erws_woocommerce_subscriptions_remind_csd_scheduled_activation_notification', array( $this, 'trigger' ), 10, 2 );

		// Call parent constructor
		parent::__construct();
	}

	/**
	 * Gets the reminder slug.
	 * 
	 * @return string
	 */
	public function get_reminder_slug() {
		return 'csd_scheduled_activation';
	}

	/**
	 * Gets the subject description.
	 * 
	 * @since 2.3.1
	 * @return string
	 */
	public function get_subject_description() {
		/* translators: %s: email subject */
		return sprintf( __( 'This controls the email subject line. Leave blank to use the default subject: %s.', 'email-reminders-for-woocommerce-subscriptions' ), '<code>' . $this->subject . '</code>' );
	}

	/**
	 * Gets the heading description.
	 * 
	 * @since 2.3.1
	 * @return string
	 */
	public function get_heading_description() {
		/* translators: %s: email heading */
		return sprintf( __( 'This controls the main heading contained within the email notification. Leave blank to use the default heading: <code>%s</code>.', 'email-reminders-for-woocommerce-subscriptions' ), $this->heading );
	}

	/**
	 * Gets the custom template sending interval description.
	 * 
	 * @since 2.3.1
	 * @return string
	 */
	public function get_custom_template_sending_interval_description() {
		return __( 'day(s) before activation date', 'email-reminders-for-woocommerce-subscriptions' );
	}

	/**
	 * Gets the default content.
	 *
	 * @since 2.3.1
	 * @return string
	 */
	public function get_default_content() {
		return "<p>Hi {customer_first_name},</p>
            <p>This is a reminder that your subscription will become active on <code>{start_date}</code>. Your subscription's details are as follows:</p>
            {subscription_details} 
            {customer_details}
            ";
	}

	/**
	 * Default content to show below main email content.
	 *
	 * @return string
	 */
	public function get_default_additional_content() {
		return __( 'Thanks for shopping with us.', 'email-reminders-for-woocommerce-subscriptions' );
	}

	/**
	 * Return if custom template is supported.
	 *
	 * @since 2.3.1
	 * @return bool
	 */
	public function supports_custom_template() {
		return true;
	}

	/**
	 * Gets the custom template supported shortcodes.
	 *
	 * @since 2.3.1
	 * @return array
	 */
	public function get_custom_template_supported_shortcodes() {
		return $this->custom_template_shortcodes;
	}

	/**
	 * Prepare the custom template shortcodes.
	 */
	protected function prepare_custom_template_shortcodes() {
		$user = new WP_User( $this->object->get_user_id() );

		$this->custom_template_shortcodes[ '{customer_name}' ]         = ucfirst( $user->display_name );
		$this->custom_template_shortcodes[ '{customer_first_name}' ]   = ucfirst( $user->first_name );
		$this->custom_template_shortcodes[ '{customer_last_name}' ]    = ucfirst( $user->last_name );
		$this->custom_template_shortcodes[ '{view_subscription_url}' ] = '<a href="' . esc_url( $this->object->get_view_order_url() ) . '">#' . esc_html( $this->object->get_order_number() ) . '</a>';
		$this->custom_template_shortcodes[ '{start_date}' ]            = $this->object->get_time( 'start' ) > 0 ? date_i18n( wc_date_format(), $this->object->get_time( 'start', 'site' ) ) : '';

		ob_start();
		WC_Subscriptions_Email::order_details( $this->object, false, 'plain' === $this->get_email_type(), $this );
		$this->custom_template_shortcodes[ '{subscription_details}' ] = ob_get_clean();

		ob_start();
		wc_get_template( 'emails/email-addresses.php', array( 'order' => $this->object, 'sent_to_admin' => false ) );
		$this->custom_template_shortcodes[ '{customer_details}' ] = ob_get_clean();
	}

	/**
	 * Get email subject.
	 *
	 * @return string
	 */
	public function get_subject() {
		$subject = $this->get_option( 'subject', $this->get_default_subject() );

		if ( $this->supports_custom_template() && asp_erws_is_custom_template( $this->custom_template_id ) ) {
			$custom_template = asp_erws_get_custom_template( $this->custom_template_id );

			if ( ! empty( $custom_template->get_email_subject() ) ) {
				$subject = $custom_template->get_email_subject();
			}
		}

		/**
		 * Get the email subject.
		 * 
		 * @param string $subject
		 * @param object $object
		 * @param WC_Email $this
		 * @since 2.3.1
		 */
		return apply_filters( 'woocommerce_email_subject_' . $this->id, $this->format_string( $subject ), $this->object, $this );
	}

	/**
	 * Get email heading.
	 *
	 * @return string
	 */
	public function get_heading() {
		$heading = $this->get_option( 'heading', $this->get_default_heading() );

		if ( $this->supports_custom_template() && asp_erws_is_custom_template( $this->custom_template_id ) ) {
			$custom_template = asp_erws_get_custom_template( $this->custom_template_id );

			if ( ! empty( $custom_template->get_email_heading() ) ) {
				$heading = $custom_template->get_email_heading();
			}
		}

		/**
		 * Get the email heading.
		 * 
		 * @param string $subject
		 * @param object $object
		 * @param WC_Email $this
		 * @since 2.3.1
		 */
		return apply_filters( 'woocommerce_email_heading_' . $this->id, $this->format_string( $heading ), $this->object, $this );
	}

	/**
	 * Get email content.
	 *
	 * @return string
	 */
	public function get_content() {
		$content = parent::get_content();

		if ( $this->supports_custom_template() && asp_erws_is_custom_template( $this->custom_template_id ) ) {
			$custom_template = asp_erws_get_custom_template( $this->custom_template_id );

			if ( ! empty( $custom_template->get_email_content() ) ) {
				$this->prepare_custom_template_shortcodes();

				ob_start();
				wc_get_template( 'emails/email-header.php', array( 'email_heading' => $this->get_heading() ) );
				$email_header = ob_get_clean();

				ob_start();
				wc_get_template( 'emails/email-footer.php' );
				$email_footer = ob_get_clean();

				if ( 'plain' === $this->get_email_type() ) {
					$content = wp_strip_all_tags( $email_header );
					$content .= str_replace( array_keys( $this->custom_template_shortcodes ), array_values( $this->custom_template_shortcodes ), wp_strip_all_tags( $custom_template->get_email_content() ) );
					$content .= wp_strip_all_tags( $email_footer );
					$content = wordwrap( $content, 70 );
				} else {
					$content = $email_header;
					$content .= str_replace( array_keys( $this->custom_template_shortcodes ), array_values( $this->custom_template_shortcodes ), $custom_template->get_email_content() );
					$content .= $email_footer;
				}
			}
		}

		return $content;
	}

	/**
	 * Maybe adds the WPML translation support.
	 * 
	 * @since 1.8.2
	 */
	public function maybe_add_wpml_switch_language() {
		/**
		 * Adds the WPML translation support.
		 * 
		 * @since 1.8.2
		 */
		do_action( 'wpml_switch_language', $this->object->get_meta( 'wpml_language' ) );
	}

	/**
	 * Trigger the sending of this email.
	 *
	 * @param WC_Subscription $subscription Subscription object.
	 * @param int $remind_day
	 */
	public function trigger( $subscription, $remind_day = 1 ) {
		if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
			$subscription = wcs_get_subscription( $subscription );
		}

		if ( is_a( $subscription, 'WC_Subscription' ) ) {
			$this->object             = $subscription;
			$this->recipient          = $this->object->get_billing_email();
			$this->remind_day         = $remind_day;
			$this->custom_template_id = asp_erws_get_custom_template_for_subscription( $this->object, $this->get_reminder_slug(), $remind_day );
		}

		$recipient = $this->get_recipient();
		if ( ! $this->is_enabled() || ! $recipient ) {
			return;
		}

		$this->maybe_add_wpml_switch_language();

		$sent = $this->send( $recipient, $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );

		/**
		 * Check if the email is sent successful.
		 * 
		 * @since 1.0.0
		 */
		do_action( 'asp_erws_is_email_sent_success', $sent, $subscription, $this );
	}

	/**
	 * Get valid recipients.
	 *
	 * @return string
	 */
	public function get_recipient() {
		// Other settings.
		$other_recipient = $this->get_option( 'recipient' );

		if ( ! empty( $other_recipient ) ) {
			$this->recipient .= ', ';
			$this->recipient .= $other_recipient;
		}

		return parent::get_recipient();
	}

	/**
	 * Get content HTML.
	 *
	 * @return string
	 */
	public function get_content_html() {
		return wc_get_template_html( $this->template_html, array(
			'subscription'       => $this->object,
			'email_heading'      => $this->get_heading(),
			'additional_content' => is_callable( array( $this, 'get_additional_content' ) ) ? $this->get_additional_content() : '',
			'sent_to_admin'      => false,
			'plain_text'         => false,
			'email'              => $this,
				), '', $this->template_base );
	}

	/**
	 * Get content plain.
	 *
	 * @return string
	 */
	public function get_content_plain() {
		return wc_get_template_html( $this->template_plain, array(
			'subscription'       => $this->object,
			'email_heading'      => $this->get_heading(),
			'additional_content' => is_callable( array( $this, 'get_additional_content' ) ) ? $this->get_additional_content() : '',
			'sent_to_admin'      => false,
			'plain_text'         => true,
			'email'              => $this,
				), '', $this->template_base );
	}

	/**
	 * Initialize settings form fields.
	 */
	public function init_form_fields() {
		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'email-reminders-for-woocommerce-subscriptions' ), '<code>' . esc_html( implode( '</code>, <code>', array_keys( $this->placeholders ) ) ) . '</code>' );
		$this->form_fields = array(
			'enabled'            => array(
				'title'   => __( 'Enable/Disable', 'email-reminders-for-woocommerce-subscriptions' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'email-reminders-for-woocommerce-subscriptions' ),
				'default' => 'yes',
			),
			'recipient'          => array(
				'title'       => __( 'Recipient(s)', 'email-reminders-for-woocommerce-subscriptions' ),
				'type'        => 'text',
				'description' => __( 'Enter recipients (comma separated) for this email. Defaults to buyer email.', 'email-reminders-for-woocommerce-subscriptions' ),
				'placeholder' => '',
				'default'     => '',
				'desc_tip'    => true,
			),
			'subject'            => array(
				'title'       => __( 'Subject', 'email-reminders-for-woocommerce-subscriptions' ),
				'type'        => 'text',
				'description' => $this->get_subject_description(),
				'placeholder' => $this->get_default_subject(),
				'default'     => '',
			),
			'heading'            => array(
				'title'       => __( 'Email Heading', 'email-reminders-for-woocommerce-subscriptions' ),
				'type'        => 'text',
				'description' => $this->get_heading_description(),
				'placeholder' => $this->get_default_heading(),
				'default'     => '',
			),
			'additional_content' => array(
				'title'       => __( 'Additional content', 'email-reminders-for-woocommerce-subscriptions' ),
				'description' => __( 'Text to appear below the main email content.', 'email-reminders-for-woocommerce-subscriptions' ) . ' ' . $placeholder_text,
				'css'         => 'width:400px; height: 75px;',
				'placeholder' => __( 'N/A', 'email-reminders-for-woocommerce-subscriptions' ),
				'type'        => 'textarea',
				'default'     => $this->get_default_additional_content(),
				'desc_tip'    => true,
			),
			'email_type'         => array(
				'title'       => __( 'Email type', 'email-reminders-for-woocommerce-subscriptions' ),
				'type'        => 'select',
				'description' => __( 'Choose which format of email to send.', 'email-reminders-for-woocommerce-subscriptions' ),
				'default'     => 'html',
				'class'       => 'email_type',
				'options'     => array(
					'plain'     => _x( 'Plain text', 'email type', 'email-reminders-for-woocommerce-subscriptions' ),
					'html'      => _x( 'HTML', 'email type', 'email-reminders-for-woocommerce-subscriptions' ),
					'multipart' => _x( 'Multipart', 'email type', 'email-reminders-for-woocommerce-subscriptions' ),
				),
			),
		);
	}
}

return new ASP_ERWS_Email_Customer_Subscription_CSD_Scheduled_Activation_Reminder();
