<?php

defined( 'ABSPATH' ) || exit;

require_once __DIR__ . '/vendor/asp-erws-external-functions.php';
require_once __DIR__ . '/class-asp-erws-autoloader.php';

/**
 * Main Email Reminders for WooCommerce Subscriptions Class.
 *
 * @class ASP_Email_Reminders_For_Woocommerce_Subscriptions
 */
final class ASP_Email_Reminders_For_Woocommerce_Subscriptions {

	/**
	 * The plugin's autoloader instance.
	 *
	 * @var ASP_ERWS_Autoloader
	 */
	protected $autoloader = null;

	/**
	 * The single instance of the class.
	 */
	protected static $instance = null;

	/**
	 * ASP_Email_Reminders_For_Woocommerce_Subscriptions constructor.
	 */
	public function __construct() {
		$this->autoloader = new ASP_ERWS_Autoloader( dirname( __DIR__ ) );
		$this->autoloader->register();

		$this->includes();
		$this->init_hooks();

		/**
		 * Fire after plugin is loaded.
		 * 
		 * @since 1.0.0
		 */
		do_action( 'asp_erws_loaded' );
	}

	/**
	 * Cloning is forbidden.
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning is forbidden.', 'email-reminders-for-woocommerce-subscriptions' ), '1.0.0' );
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'email-reminders-for-woocommerce-subscriptions' ), '1.0.0' );
	}

	/**
	 * Main ASP_Email_Reminders_For_Woocommerce_Subscriptions Instance.
	 * Ensures only one instance of ASP_Email_Reminders_For_Woocommerce_Subscriptions is loaded or can be loaded.
	 * 
	 * @return ASP_Email_Reminders_For_Woocommerce_Subscriptions - Main instance.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public function plugin_url() {
		return untrailingslashit( plugins_url( '/', ASP_ERWS_PLUGIN_FILE ) );
	}

	/**
	 * Get the plugin path.
	 *
	 * @return string
	 */
	public function plugin_path() {
		return untrailingslashit( plugin_dir_path( ASP_ERWS_PLUGIN_FILE ) );
	}

	/**
	 * Get the plugin basename.
	 *
	 * @return string
	 */
	public function plugin_basename() {
		return plugin_basename( ASP_ERWS_PLUGIN_FILE );
	}

	/**
	 * Includes required files.
	 */
	private function includes() {
		include_once 'asp-erws-custom-template-functions.php';
		include_once 'class-asp-erws-emails.php';

		if ( is_admin() ) {
			include_once 'admin/class-asp-erws-admin.php';
		}
	}

	/**
	 * Hook into actions and filters.
	 */
	private function init_hooks() {
		add_filter( 'plugin_action_links_' . ASP_ERWS_PLUGIN_BASENAME, array( $this, 'plugin_action_links' ) );
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );
		add_action( 'init', array( $this, 'register_post_types' ) );
		add_action( 'init', array( $this, 'register_post_statuses' ) );
		add_action( 'init', array( $this, 'other_plugin_support_includes' ), 20 );
		add_action( 'action_scheduler_init', array( $this, 'action_scheduler_init' ) );
		add_filter( 'woocommerce_data_stores', array( $this, 'add_data_stores' ) );

		add_action( 'woocommerce_checkout_create_subscription', array( $this, 'subscription_created' ), 10, 4 );
		add_action( 'woocommerce_checkout_create_order_line_item', array( $this, 'subscription_item_created' ), 10, 4 );
		add_action( 'woocommerce_subscriptions_switched_item', array( $this, 'subscription_switched' ), 10, 2 );
		add_filter( 'woocommerce_hidden_order_itemmeta', array( $this, 'hidden_subscription_itemmeta' ) );

		add_action( 'wp_ajax_asp_erws_post_ordering', array( $this, 'post_ordering' ) );
		add_action( 'wp_ajax_asp_erws_custom_template_email_data', array( $this, 'custom_template_email_data' ) );
		add_action( 'wp_ajax_asp_erws_test_custom_template_email', array( $this, 'test_custom_template_email' ) );
		add_action( 'wp_ajax_asp_erws_init_existing_subscriptions_scheduler', array( $this, 'init_existing_subscriptions_scheduler' ) );
		add_action( 'woocommerce_scheduled_subscription_payment', array( $this, 'apply_product_meta_for_old_subscription' ), -5 );

		add_action( 'woocommerce_subscription_status_updated', array( $this, 'maybe_schedule_when_status_updated' ), 0, 2 );
		add_action( 'woocommerce_subscription_date_updated', array( $this, 'maybe_schedule_when_date_updated' ), 0, 2 );
		add_action( 'woocommerce_subscription_date_deleted', array( $this, 'maybe_schedule_when_date_updated' ), 0, 2 );
		add_action( 'woocommerce_subscriptions_switch_completed', array( $this, 'maybe_schedule_when_switched' ), 11 );

		add_action( 'asp_erws_scheduled_existing_subscriptions_scheduler', array( $this, 'maybe_schedule_reminders_for_existing_subscriptions' ), 10 );
		add_action( 'asp_erws_woocommerce_scheduled_subscription_trial_ending_reminder', array( $this, 'remind_trial_ending' ), 10, 2 );
		add_action( 'asp_erws_woocommerce_scheduled_subscription_auto_renewal_reminder', array( $this, 'remind_renewal' ), 10, 2 );
		add_action( 'asp_erws_woocommerce_scheduled_subscription_manual_renewal_reminder', array( $this, 'remind_renewal' ), 10, 2 );
		add_action( 'asp_erws_woocommerce_scheduled_subscription_expiration_reminder', array( $this, 'remind_expiration' ), 10, 2 );
		add_action( 'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder', array( $this, 'remind_after_expiry' ), 10, 2 );
	}

	/**
	 * Load Localization files.
	 */
	public function load_plugin_textdomain() {
		if ( function_exists( 'determine_locale' ) ) {
			$locale = determine_locale();
		} else {
			$locale = is_admin() ? get_user_locale() : get_locale();
		}

		/**
		 * Get the plugin locale.
		 * 
		 * @since 1.0.0
		 */
		$locale = apply_filters( 'plugin_locale', $locale, 'email-reminders-for-woocommerce-subscriptions' );

		unload_textdomain( 'email-reminders-for-woocommerce-subscriptions' );
		load_textdomain( 'email-reminders-for-woocommerce-subscriptions', WP_LANG_DIR . '/email-reminders-for-woocommerce-subscriptions/email-reminders-for-woocommerce-subscriptions-' . $locale . '.mo' );
		load_plugin_textdomain( 'email-reminders-for-woocommerce-subscriptions', false, dirname( $this->plugin_basename() ) . '/languages' );
	}

	/**
	 * Get reminders.
	 * 
	 * @return array
	 */
	public function get_reminders() {
		$reminders = array(
			'trial_end'      => __( 'Subscription Trial Ending Reminder', 'email-reminders-for-woocommerce-subscriptions' ),
			'manual_renewal' => __( 'Subscription Manual Renewal Reminder', 'email-reminders-for-woocommerce-subscriptions' ),
			'auto_renewal'   => __( 'Subscription Auto Renewal Reminder', 'email-reminders-for-woocommerce-subscriptions' ),
			'expiration'     => __( 'Subscription Expiration Reminder', 'email-reminders-for-woocommerce-subscriptions' ),
			'after_expiry'   => __( 'Subscription After Expiry Reminder', 'email-reminders-for-woocommerce-subscriptions' ),
		);

		/**
		 * Get reminders.
		 * 
		 * @since 2.3.1
		 */
		return apply_filters( 'asp_erws_get_reminders', $reminders );
	}

	/**
	 * Get reminder hooks.
	 * 
	 * @return array
	 */
	public function get_reminder_hooks() {
		$reminder_hooks = array(
			'asp_erws_woocommerce_scheduled_subscription_trial_ending_reminder'   => 'trial_end',
			'asp_erws_woocommerce_scheduled_subscription_auto_renewal_reminder'   => 'next_payment',
			'asp_erws_woocommerce_scheduled_subscription_manual_renewal_reminder' => 'next_payment',
			'asp_erws_woocommerce_scheduled_subscription_expiration_reminder'     => 'end',
			'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder'   => 'end',
		);

		/**
		 * Get reminder hooks.
		 * 
		 * @since 2.3.1
		 */
		return apply_filters( 'asp_erws_get_reminder_hooks', $reminder_hooks );
	}

	/**
	 * Get reminder intervals.
	 * 
	 * @return array
	 */
	public function get_reminder_intervals() {
		$reminder_intervals = array(
			'trial_end'      => get_option( 'erws_trial_ending_reminder_days' ),
			'manual_renewal' => get_option( 'erws_manual_renewal_reminder_days' ),
			'auto_renewal'   => get_option( 'erws_auto_renewal_reminder_days' ),
			'expiration'     => get_option( 'erws_expiration_reminder_days' ),
			'after_expiry'   => get_option( 'erws_after_expiry_reminder_days' ),
		);

		/**
		 * Get reminder intervals.
		 * 
		 * @since 2.3.1
		 */
		return apply_filters( 'asp_erws_get_reminder_intervals', $reminder_intervals );
	}

	/**
	 * Get the reminder class.
	 * 
	 * @return string
	 */
	public function get_reminder_email_class_from_slug( $slug ) {
		/**
		 * Get reminder classes.
		 * 
		 * @since 2.3.1
		 */
		$reminder_classes = apply_filters( 'asp_erws_get_reminder_email_classes', array(
			'trial_end'      => 'ASP_ERWS_Email_Customer_Subscription_Trial_Ending_Reminder',
			'manual_renewal' => 'ASP_ERWS_Email_Customer_Subscription_Manual_Renewal_Reminder',
			'auto_renewal'   => 'ASP_ERWS_Email_Customer_Subscription_Auto_Renewal_Reminder',
			'expiration'     => 'ASP_ERWS_Email_Customer_Subscription_Expiration_Reminder',
			'after_expiry'   => 'ASP_ERWS_Email_Customer_Subscription_After_Expiry_Reminder',
				) );

		return isset( $reminder_classes[ $slug ] ) ? $reminder_classes[ $slug ] : '';
	}

	/**
	 * Get the args to set on the scheduled action.
	 *
	 * @param string $date_type Can be 'trial_end', 'next_payment', 'expiration', 'end_of_prepaid_term' or a custom date type
	 * @param object $subscription An instance of WC_Subscription object
	 * @return array Array of name => value pairs stored against the scheduled action.
	 */
	public function get_action_args( $date_type, $subscription ) {
		/**
		 * Get scheduled action args.
		 * 
		 * @since version 1.0.0
		 */
		return apply_filters( 'woocommerce_subscriptions_scheduled_action_args', array( 'subscription_id' => $subscription->get_id() ), $date_type, $subscription );
	}

	/**
	 * Array of dates between the given dates.
	 * 
	 * @param mixed $start_time A valid date/time string
	 * @param mixed $end_time A valid date/time string
	 * @param array $days_count
	 * @param string $reminder_hook
	 * @return array
	 */
	public function get_dates( $start_time, $end_time, $days_count, $reminder_hook = 'any' ) {
		$dates = array();

		if ( ! empty( $days_count ) && is_array( $days_count ) ) {
			$start_datetime = $this->get_datetime( $start_time );
			$end_datetime   = $this->get_datetime( $end_time );

			foreach ( $days_count as $day_count ) {
				$day_count = absint( $day_count );

				if ( $day_count ) {
					if ( 'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder' === $reminder_hook ) {
						$timestamp = strtotime( "+{$day_count} days", $end_datetime->getTimestamp() );
					} else {
						$timestamp = strtotime( "-{$day_count} days", $end_datetime->getTimestamp() );
					}

					if ( $timestamp >= $start_datetime->getTimestamp() ) {
						$dates[ $day_count ] = $timestamp;
					}
				}
			}

			if ( ! $dates ) {
				if ( 'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder' === $reminder_hook ) {
					$timestamp = strtotime( '+12 hours', $end_datetime->getTimestamp() );
				} else {
					$timestamp = strtotime( '-12 hours', $end_datetime->getTimestamp() );
				}

				if ( $timestamp >= $start_datetime->getTimestamp() ) {
					$dates[ 1 ] = $timestamp;
				}
			}

			if ( $dates ) {
				$dates = array_unique( $dates );
				asort( $dates );
			}
		}

		return $dates;
	}

	/**
	 * Gets the datetime.
	 *
	 * @param string|integer $value Value of the prop.
	 * @return WC_DateTime
	 */
	public function get_datetime( $value ) {
		if ( is_a( $value, 'WC_DateTime' ) ) {
			$datetime = $value;
		} elseif ( is_numeric( $value ) ) {
			// Timestamps are handled as UTC timestamps in all cases.
			$datetime = new WC_DateTime( "@{$value}", new DateTimeZone( 'UTC' ) );
		} else {
			// Strings are defined in local WP timezone. Convert to UTC.
			$timestamp = wc_string_to_timestamp( get_gmt_from_date( gmdate( 'Y-m-d H:i:s', wc_string_to_timestamp( $value ) ) ) );
			$datetime  = new WC_DateTime( "@{$timestamp}", new DateTimeZone( 'UTC' ) );
		}

		// Set local timezone or offset.
		if ( get_option( 'timezone_string' ) ) {
			$datetime->setTimezone( new DateTimeZone( wc_timezone_string() ) );
		} else {
			$datetime->set_utc_offset( wc_timezone_offset() );
		}

		return $datetime;
	}

	/**
	 * Check if what type the array is sorted by.
	 * 
	 * @param array $array
	 * @return string
	 */
	public function is_array_sorted_by( $array ) {
		$maybe_asc_array = $array;
		sort( $maybe_asc_array );
		return ( $array === $maybe_asc_array ) ? 'ASC' : 'DESC';
	}

	/**
	 * Check if any reminder is configured.
	 * 
	 * @return bool 
	 */
	public function is_reminder_configured() {
		$reminder_intervals = $this->get_reminder_intervals();
		$configured         = false;

		foreach ( $reminder_intervals as $slug => $interval ) {
			if ( ! empty( $interval ) ) {
				$configured = true;
				break;
			}
		}

		return $configured;
	}

	/**
	 * Is valid to schedule the given reminder?
	 * 
	 * @param WC_Subscription $subscription
	 * @param string $reminder_slug
	 * @return bool
	 */
	public function can_schedule_reminder( $subscription, $reminder_slug ) {
		$exclude_reminder_emails = $subscription->get_meta( '_asp_erws_exclude_reminder_emails', true, 'edit' );

		if ( '' === $exclude_reminder_emails || ! is_array( $exclude_reminder_emails ) ) {
			$can = true;
		} else {
			$can = in_array( $reminder_slug, $exclude_reminder_emails ) ? false : true;
		}

		return $can;
	}

	/**
	 * Schedule the multiple reminders before the end time.
	 * 
	 * @param mixed $end_time
	 * @param string $hook
	 * @param mixed $days_to_remind
	 * @param array $action_args
	 */
	public function schedule_reminders( $end_time, $hook, $days_to_remind, $action_args ) {
		if ( empty( $days_to_remind ) ) {
			return;
		}

		$days_to_remind = ! is_array( $days_to_remind ) ? array_map( 'trim', explode( ',', $days_to_remind ) ) : $days_to_remind;
		$days_to_remind = array_map( 'absint', $days_to_remind );
		$days_to_remind = $this->get_dates( time(), $end_time, $days_to_remind, $hook );

		if ( ! empty( $days_to_remind ) ) {
			foreach ( $days_to_remind as $remind_day => $timestamp ) {
				$action_args[ 'remind_day' ] = $remind_day;

				if ( as_next_scheduled_action( $hook, $action_args ) !== $timestamp ) {
					as_schedule_single_action( $timestamp, $hook, $action_args );
				}
			}
		}
	}

	/**
	 * Schedule the actions for the date type given.
	 * 
	 * @param WC_Subscription $subscription
	 * @param int $timestamp
	 * @param string $date_type Can be 'trial_end', 'next_payment', 'end', 'end_of_prepaid_term' or a custom date type
	 * @param array $action_args
	 * @param bool $force
	 */
	private function schedule_actions( $subscription, $timestamp, $date_type, $action_args, $force = false ) {
		if ( ! $timestamp ) {
			return;
		}

		switch ( $date_type ) {
			case 'trial_end':
				if ( as_next_scheduled_action( 'woocommerce_scheduled_subscription_trial_end', $action_args ) !== $timestamp || $force ) {
					// Clear the respective schedules first.
					$this->unschedule_all_actions_by_date_type( 'trial_end', $action_args );

					if ( $this->can_schedule_reminder( $subscription, 'trial_end' ) ) {
						$this->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_trial_ending_reminder', get_option( 'erws_trial_ending_reminder_days' ), $action_args );
					}
				}
				break;
			case 'next_payment':
				if ( as_next_scheduled_action( 'woocommerce_scheduled_subscription_payment', $action_args ) !== $timestamp || $force ) {
					// Clear the respective schedules first.
					$this->unschedule_all_actions_by_date_type( 'next_payment', $action_args );

					if ( $subscription->is_manual() ) {
						if ( $this->can_schedule_reminder( $subscription, 'manual_renewal' ) ) {
							$this->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_manual_renewal_reminder', get_option( 'erws_manual_renewal_reminder_days' ), $action_args );
						}
					} elseif ( $this->can_schedule_reminder( $subscription, 'auto_renewal' ) ) {
						$this->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_auto_renewal_reminder', get_option( 'erws_auto_renewal_reminder_days' ), $action_args );
					}
				}
				break;
			case 'end':
				if ( $subscription->has_status( 'expired' ) && $this->can_schedule_reminder( $subscription, 'after_expiry' ) ) {
					$this->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder', get_option( 'erws_after_expiry_reminder_days' ), $action_args );
					break;
				}

				if ( as_next_scheduled_action( 'woocommerce_scheduled_subscription_expiration', $action_args ) !== $timestamp || $force ) {
					// Clear the respective schedules first.
					$this->unschedule_all_actions_by_hook( 'asp_erws_woocommerce_scheduled_subscription_expiration_reminder', $action_args );
					$this->unschedule_all_actions_by_hook( 'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder', $action_args );

					if ( $this->can_schedule_reminder( $subscription, 'expiration' ) ) {
						$this->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_expiration_reminder', get_option( 'erws_expiration_reminder_days' ), $action_args );
					}

					if ( $subscription->has_status( 'active' ) && $this->can_schedule_reminder( $subscription, 'after_expiry' ) ) {
						$this->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_after_expiry_reminder', get_option( 'erws_after_expiry_reminder_days' ), $action_args );
					}
				}
				break;
		}
	}

	/**
	 * Unschedule all actions by hook in bulk.
	 * 
	 * @param string $hook
	 * @param array $action_args
	 */
	public function unschedule_all_actions_by_hook( $hook, $action_args ) {
		$scheduled_actions = as_get_scheduled_actions( array( 'hook' => $hook, 'status' => 'pending', 'per_page' => -1 ) );

		foreach ( $scheduled_actions as $action ) {
			$args = $action->get_args();

			if ( $action_args[ 'subscription_id' ] == $args[ 'subscription_id' ] ) {
				as_unschedule_all_actions( $hook, $args );
			}
		}
	}

	/**
	 * Unschedule all actions by date type in bulk.
	 * 
	 * @param string $date_type
	 * @param array $action_args
	 */
	public function unschedule_all_actions_by_date_type( $date_type, $action_args ) {
		foreach ( $this->get_reminder_hooks() as $hook => $value ) {
			if ( $value === $date_type ) {
				$this->unschedule_all_actions_by_hook( $hook, $action_args );
			}
		}
	}

	/**
	 * Unschedule all actions in bulk.
	 * 
	 * @param object $subscription An instance of a WC_Subscription object
	 * @param array $exclude_hooks
	 */
	public function unschedule_all_actions( $subscription, $exclude_hooks = array() ) {
		foreach ( $this->get_reminder_hooks() as $hook => $date_type ) {
			if ( in_array( $hook, ( array ) $exclude_hooks ) ) {
				continue;
			}

			$this->unschedule_all_actions_by_hook( $hook, $this->get_action_args( $date_type, $subscription ) );
		}
	}

	/**
	 * Show action links on the plugin screen.
	 *
	 * @param   mixed $links Plugin Action links
	 * @return  array
	 */
	public function plugin_action_links( $links ) {
		$setting_page_link = '<a  href="' . admin_url( 'admin.php?page=wc-settings&tab=subscriptions#erws_reminders-description' ) . '">' . esc_html__( 'Settings', 'email-reminders-for-woocommerce-subscriptions' ) . '</a>';
		array_unshift( $links, $setting_page_link );
		return $links;
	}

	/**
	 * Action Scheduler is loaded.
	 */
	public function action_scheduler_init() {
		// Remove legacy schedules.
		as_unschedule_all_actions( 'asp_erws_scheduled_existing_subscriptions_scheduler' );
	}

	/**
	 * Register core post types.
	 */
	public function register_post_types() {
		if ( ! post_type_exists( 'asp_erws_custom_tmpl' ) ) {
			register_post_type( 'asp_erws_custom_tmpl', array(
				'labels'              => array(
					'name'               => __( 'Manage Custom Subscription Email Reminder Templates', 'email-reminders-for-woocommerce-subscriptions' ),
					'singular_name'      => __( 'Custom Subscription Email Reminder Template', 'email-reminders-for-woocommerce-subscriptions' ),
					'menu_name'          => __( 'Custom Subscription Email Reminder Templates', 'email-reminders-for-woocommerce-subscriptions' ),
					'add_new'            => __( 'Add template', 'email-reminders-for-woocommerce-subscriptions' ),
					'add_new_item'       => __( 'Add custom subscription email reminder template', 'email-reminders-for-woocommerce-subscriptions' ),
					'new_item'           => __( 'New custom email reminder template', 'email-reminders-for-woocommerce-subscriptions' ),
					'edit_item'          => __( 'Edit custom subscription email reminder template', 'email-reminders-for-woocommerce-subscriptions' ),
					'view_item'          => __( 'View custom email reminder template', 'email-reminders-for-woocommerce-subscriptions' ),
					'search_items'       => __( 'Search templates', 'email-reminders-for-woocommerce-subscriptions' ),
					'not_found'          => __( 'No templates found.', 'email-reminders-for-woocommerce-subscriptions' ),
					'not_found_in_trash' => __( 'No templates found in Trash.', 'email-reminders-for-woocommerce-subscriptions' ),
				),
				'description'         => __( 'This is where store custom subscription reminder email templates are stored.', 'email-reminders-for-woocommerce-subscriptions' ),
				'public'              => false,
				'capability_type'     => 'post',
				'show_ui'             => true,
				'publicly_queryable'  => false,
				'exclude_from_search' => true,
				'show_in_menu'        => false,
				'hierarchical'        => false,
				'show_in_nav_menus'   => false,
				'show_in_admin_bar'   => false,
				'rewrite'             => false,
				'query_var'           => false,
				'supports'            => array( 'title' ),
				'has_archive'         => false,
				'map_meta_cap'        => true,
			) );
		}

		/**
		 * After register post type
		 * 
		 * @since 1.8.0
		 */
		do_action( 'asp_erws_after_register_post_type' );
	}

	/**
	 * Register our post statuses.
	 */
	public function register_post_statuses() {
		$our_statuses = array(
			'asp-' => array(
				'active'   => array(
					'label'                     => _x( 'Active', 'custom email reminder template status name', 'email-reminders-for-woocommerce-subscriptions' ),
					'public'                    => true,
					'exclude_from_search'       => false,
					'show_in_admin_all_list'    => true,
					'show_in_admin_status_list' => true,
					/* translators: %s: status name */
					'label_count'               => _n_noop( 'Active <span class="count">(%s)</span>', 'Active <span class="count">(%s)</span>' ),
				),
				'inactive' => array(
					'label'                     => _x( 'Inactive', 'custom email reminder template status name', 'email-reminders-for-woocommerce-subscriptions' ),
					'public'                    => true,
					'exclude_from_search'       => false,
					'show_in_admin_all_list'    => true,
					'show_in_admin_status_list' => true,
					/* translators: %s: status name */
					'label_count'               => _n_noop( 'Inactive <span class="count">(%s)</span>', 'Inactive <span class="count">(%s)</span>' ),
				),
			),
		);

		foreach ( $our_statuses as $prefix => $statuses ) {
			foreach ( $statuses as $status => $args ) {
				register_post_status( $prefix . $status, $args );
			}
		}
	}

	/**
	 * Include classes for other plugins support.
	 */
	public function other_plugin_support_includes() {
		include_once 'compat/class-asp-erws-compat-custom-start-date.php';

		if ( class_exists( 'Scheduled_Woocommerce_Subscription_Status' ) ) {
			ASP_SSWS_Compat_CSD_For_WCS::init();
		}
	}

	/**
	 * Add our data stores to WC.
	 * 
	 * @param array $data_stores
	 * @return array
	 */
	public function add_data_stores( $data_stores ) {
		$data_stores[ 'asp_erws_custom_template' ] = 'ASP_ERWS_Custom_Template_Data_Store_CPT';
		return $data_stores;
	}

	/**
	 * Ajax request handling for post ordering.
	 */
	public function post_ordering() {
		global $wpdb;

		$posted = asp_erws_get_global_var( 'post' );
		if ( ! isset( $posted[ 'id' ] ) ) {
			wp_die( -1 );
		}

		$sorting_id  = absint( $posted[ 'id' ] );
		$post_type   = get_post_type( $sorting_id );
		$previd      = absint( isset( $posted[ 'previd' ] ) ? $posted[ 'previd' ] : 0 );
		$nextid      = absint( isset( $posted[ 'nextid' ] ) ? $posted[ 'nextid' ] : 0 );
		$menu_orders = wp_list_pluck( $wpdb->get_results( $wpdb->prepare( "SELECT ID, menu_order FROM {$wpdb->posts} WHERE post_type=%s ORDER BY menu_order ASC, post_title ASC", esc_sql( $post_type ) ) ), 'menu_order', 'ID' );
		$index       = 0;

		foreach ( $menu_orders as $id => $menu_order ) {
			$id = absint( $id );

			if ( $sorting_id === $id ) {
				continue;
			}
			if ( $nextid === $id ) {
				$index++;
			}
			$index++;
			$menu_orders[ $id ] = $index;
			$wpdb->update( $wpdb->posts, array( 'menu_order' => $index ), array( 'ID' => $id ) );
		}

		if ( isset( $menu_orders[ $previd ] ) ) {
			$menu_orders[ $sorting_id ] = $menu_orders[ $previd ] + 1;
		} elseif ( isset( $menu_orders[ $nextid ] ) ) {
			$menu_orders[ $sorting_id ] = $menu_orders[ $nextid ] - 1;
		} else {
			$menu_orders[ $sorting_id ] = 0;
		}

		$wpdb->update( $wpdb->posts, array( 'menu_order' => $menu_orders[ $sorting_id ] ), array( 'ID' => $sorting_id ) );
		wp_send_json( $menu_orders );
	}

	/**
	 * Output the custom template email data.
	 */
	public function custom_template_email_data() {
		check_ajax_referer( 'asp-erws-custom-template-email-data', 'security' );

		try {
			$posted          = asp_erws_get_global_var( 'post' );
			$template_id     = isset( $posted[ 'template_id' ] ) ? absint( wp_unslash( $posted[ 'template_id' ] ) ) : 0;
			$email_slug      = isset( $posted[ 'email_slug' ] ) ? sanitize_title( wp_unslash( $posted[ 'email_slug' ] ) ) : '';
			$custom_template = asp_erws_get_custom_template( $template_id );

			if ( ! $custom_template ) {
				throw new Exception( esc_html__( 'Invalid template.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			$selected_email_class = $this->get_reminder_email_class_from_slug( $email_slug );
			$emails               = WC()->mailer()->get_emails();

			if ( empty( $selected_email_class ) || empty( $emails[ $selected_email_class ] ) ) {
				throw new Exception( esc_html__( 'Invalid email ID.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			$selected_reminder_email = $emails[ $selected_email_class ];

			if ( ! $selected_reminder_email->supports_custom_template() ) {
				throw new Exception( esc_html__( 'Invalid email supported.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			$email_shortcodes = array();
			$email_field_atts = array();

			foreach ( $selected_reminder_email->get_custom_template_supported_shortcodes() as $shortcode => $value ) {
				switch ( $shortcode ) {
					case '{customer_name}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the customer name.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{customer_first_name}':
						$email_shortcodes[ $shortcode ] = __( "Displays the customer's first name.", 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{customer_last_name}':
						$email_shortcodes[ $shortcode ] = __( "Displays the customer's last name.", 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{customer_details}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the customer details.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{view_subscription_url}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the view subscription url.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{resubscribe_url}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the resubscribe url.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{start_date}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the start date.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{trial_end_date}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the trial end date.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{next_payment_date}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the next payment date.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{end_date}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the subscription end date.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
					case '{subscription_details}':
						$email_shortcodes[ $shortcode ] = __( 'Displays the subscription details.', 'email-reminders-for-woocommerce-subscriptions' );
						break;
				}
			}

			$email_field_atts[ 'placeholder' ][ 'subject' ]                          = $selected_reminder_email->get_default_subject();
			$email_field_atts[ 'placeholder' ][ 'heading' ]                          = $selected_reminder_email->get_default_heading();
			$email_field_atts[ 'description' ][ 'subject' ]                          = $selected_reminder_email->get_subject_description();
			$email_field_atts[ 'description' ][ 'heading' ]                          = $selected_reminder_email->get_heading_description();
			$email_field_atts[ 'description' ][ 'custom_template_sending_interval' ] = $selected_reminder_email->get_custom_template_sending_interval_description();
			$email_field_atts[ 'value' ][ 'content' ]                                = $selected_reminder_email->get_default_content();

			if ( $email_slug !== $custom_template->get_email_slug() ) {
				$custom_template->set_email_subject( '' );
				$custom_template->set_email_heading( '' );
				$custom_template->set_email_content( $selected_reminder_email->get_default_content() );

				$custom_template->set_criteria_user_filter( 'all-users' );
				$custom_template->set_criteria_users( '' );
				$custom_template->set_criteria_user_roles( '' );

				$custom_template->set_criteria_product_filter( 'all-products' );
				$custom_template->set_criteria_products( '' );
				$custom_template->set_criteria_product_cats( '' );
			} else {
				$email_content = $custom_template->get_email_content();
				$email_content = ! empty( $email_content ) ? $email_content : $selected_reminder_email->get_default_content();
				$custom_template->set_email_content( $email_content );
			}

			ob_start();
			include 'admin/meta-boxes/views/html-custom-template-data-general.php';

			_WP_Editors::editor_js();
			_WP_Editors::enqueue_scripts();
			print_footer_scripts();

			$general_section_html = ob_get_clean();

			ob_start();
			include 'admin/meta-boxes/views/html-custom-template-data-criteria.php';
			$criteria_section_html = ob_get_clean();

			ob_start();
			include 'admin/meta-boxes/views/html-custom-template-shortcodes.php';
			$shortcodes_section_html = ob_get_clean();

			wp_send_json_success( array(
				'general_section_html'    => $general_section_html,
				'criteria_section_html'   => $criteria_section_html,
				'shortcodes_section_html' => $shortcodes_section_html,
			) );
		} catch ( Exception $e ) {
			wp_die();
		}
	}

	/**
	 * Output the custom template test email.
	 */
	public function test_custom_template_email( $param ) {
		check_ajax_referer( 'asp-erws-test-custom-template-email', 'security' );

		try {
			$posted                 = asp_erws_get_global_var( 'post' );
			$template_id            = isset( $posted[ 'template_id' ] ) ? absint( wp_unslash( $posted[ 'template_id' ] ) ) : 0;
			$subscription_id        = isset( $posted[ 'subscription_id' ] ) ? absint( wp_unslash( $posted[ 'subscription_id' ] ) ) : 0;
			$email_sending_interval = isset( $posted[ 'email_sending_interval' ] ) ? absint( wp_unslash( $posted[ 'email_sending_interval' ] ) ) : 0;
			$email_slug             = isset( $posted[ 'email_slug' ] ) ? sanitize_title( wp_unslash( $posted[ 'email_slug' ] ) ) : '';
			$email_recipients       = isset( $posted[ 'email_recipients' ] ) ? wc_clean( wp_unslash( $posted[ 'email_recipients' ] ) ) : '';
			$custom_template        = asp_erws_get_custom_template( $template_id );
			$subscription           = wcs_get_subscription( $subscription_id );

			if ( ! $custom_template ) {
				throw new Exception( esc_html__( 'Invalid template.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			$selected_email_class = $this->get_reminder_email_class_from_slug( $email_slug );
			$emails               = WC()->mailer()->get_emails();

			if ( empty( $selected_email_class ) || empty( $emails[ $selected_email_class ] ) ) {
				throw new Exception( esc_html__( 'Invalid email ID.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			$email = $emails[ $selected_email_class ];

			if ( ! $email->supports_custom_template() ) {
				throw new Exception( esc_html__( 'Invalid email supported.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			if ( ! $subscription ) {
				throw new Exception( esc_html__( 'No subscription found.', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			$recipients = array_map( 'trim', explode( ',', $email_recipients ) );
			$recipients = array_filter( $recipients, 'is_email' );
			$recipients = implode( ', ', $recipients );

			$email->object             = $subscription;
			$email->recipient          = $recipients;
			$email->remind_day         = $email_sending_interval;
			$email->custom_template_id = $template_id;

			$email->maybe_add_wpml_switch_language();

			if ( ! $email->send( $email->recipient, $email->get_subject(), $email->get_content(), $email->get_headers(), $email->get_attachments() ) ) {
				throw new Exception( esc_html__( 'Failed to send the email !!', 'email-reminders-for-woocommerce-subscriptions' ) );
			}

			wp_send_json_success( array( 'message' => esc_html__( 'Test email has been sent successfully !!', 'email-reminders-for-woocommerce-subscriptions' ) ) );
		} catch ( Exception $e ) {
			wp_send_json_error( array( 'error' => esc_html( $e->getMessage() ) ) );
		}
	}

	/**
	 * Init the scheduler for existing subscriptions.
	 */
	public function init_existing_subscriptions_scheduler() {
		check_ajax_referer( 'asp-erws-existing-subscriptions-scheduler', 'security' );

		$posted    = asp_erws_get_global_var( 'post' );
		$intervals = isset( $posted[ 'intervals' ] ) ? wp_parse_args( $posted[ 'intervals' ] ) : array();

		if ( ! empty( $intervals ) ) {
			foreach ( $intervals as $meta_key => $interval ) {
				update_option( "$meta_key", wc_clean( wp_unslash( $interval ) ) );
			}
		}

		if ( ! wp_next_scheduled( 'asp_erws_scheduled_existing_subscriptions_scheduler' ) ) {
			wp_schedule_single_event( time() + 5, 'asp_erws_scheduled_existing_subscriptions_scheduler' );
		}

		wp_die();
	}

	/**
	 * Maybe apply product meta for old subscription.
	 * 
	 * @param int $subscription_id
	 */
	public function apply_product_meta_for_old_subscription( $subscription_id ) {
		$subscription = wcs_get_subscription( $subscription_id );
		if ( ! $subscription ) {
			return;
		}

		$items = $subscription->get_items( 'line_item' );
		if ( ! empty( $items ) ) {
			foreach ( $items as $item ) {
				$product = $item->get_product();

				if ( $product ) {
					$subscription->update_meta_data( '_asp_erws_exclude_reminder_emails', WC_Subscriptions_Product::get_meta_data( $product, '_asp_erws_exclude_reminder_emails', array() ) );
					$subscription->save();
				}
			}
		}
	}

	/**
	 * Update subscription meta.
	 * 
	 * @param WC_Subscription $subscription
	 * @param array $posted_data
	 * @param WC_Order $order
	 * @param WC_Cart $cart
	 */
	public function subscription_created( $subscription, $posted_data, $order, $cart ) {
		$subscription->update_meta_data( '_asp_erws_exclude_reminder_emails', wcs_cart_pluck( $cart, '_asp_erws_exclude_reminder_emails', array() ) );
		$subscription->save();
	}

	/**
	 * Update subscription meta.
	 * 
	 * @param WC_Order_Item $item
	 * @param string $cart_item_key
	 * @param array $cart_item
	 * @param WC_Subscription $subscription
	 */
	public function subscription_item_created( $item, $cart_item_key, $cart_item, $subscription ) {
		if ( wcs_is_subscription( $subscription ) ) {
			$subscription->update_meta_data( '_asp_erws_exclude_reminder_emails', WC_Subscriptions_Product::get_meta_data( $cart_item[ 'data' ], '_asp_erws_exclude_reminder_emails', array() ) );
			$subscription->save();
		}
	}

	/**
	 * Update subscription meta.
	 * 
	 * @param WC_Subscription $subscription
	 * @param WC_Order_Item $new_order_item
	 */
	public function subscription_switched( $subscription, $new_order_item ) {
		$subscription->update_meta_data( '_asp_erws_exclude_reminder_emails', $new_order_item->get_meta( '_asp_erws_exclude_reminder_emails' ) );
		$subscription->save();
	}

	/**
	 * To hide subscription meta.
	 * 
	 * @return array
	 */
	public function hidden_subscription_itemmeta( $hidden ) {
		$hidden[] = '_asp_erws_exclude_reminder_emails';
		return $hidden;
	}

	/**
	 * When a subscription's status is updated, maybe schedule some events.
	 *
	 * @param object $subscription An instance of a WC_Subscription object
	 * @param string $new_status
	 * @param string $old_status
	 */
	public function maybe_schedule_when_status_updated( $subscription, $new_status ) {
		switch ( $new_status ) {
			case 'active':
				$this->maybe_schedule_when_date_updated( $subscription, 'trial_end' );
				$this->maybe_schedule_when_date_updated( $subscription, 'next_payment' );
				$this->maybe_schedule_when_date_updated( $subscription, 'end' );
				break;
			case 'expired':
				$this->maybe_schedule_when_date_updated( $subscription, 'end' );
				break;
			default:
				$this->unschedule_all_actions( $subscription );
				break;
		}

		/**
		 * After reminders may be scheduled.
		 * 
		 * @since 2.3.1
		 */
		do_action( 'asp_erws_after_reminders_maybe_scheduled_for_status', $new_status, $subscription );
	}

	/**
	 * When a subscription's date is updated, maybe schedule some events.
	 *
	 * @param WC_Subscription $subscription
	 * @param string $date_type Can be 'trial_end', 'next_payment', 'end', 'end_of_prepaid_term' or a custom date type
	 */
	public function maybe_schedule_when_date_updated( $subscription, $date_type ) {
		if ( ! $subscription->has_status( 'active' ) ) {
			$this->unschedule_all_actions( $subscription );
		}

		/**
		 * Run the below code only when the subscription is either active or expired. 
		 */
		if ( $subscription->has_status( array( 'active', 'expired' ) ) ) {
			$timestamp   = $subscription->get_time( $date_type );
			$action_args = $this->get_action_args( $date_type, $subscription );

			if ( ! $timestamp ) {
				$this->unschedule_all_actions_by_date_type( $date_type, $action_args );
				return;
			}

			$this->schedule_actions( $subscription, $timestamp, $date_type, $action_args, true );
		}

		/**
		 * After reminders may be scheduled.
		 * 
		 * @since 2.3.1
		 */
		do_action( 'asp_erws_after_reminders_maybe_scheduled_for_date_type', $date_type, $subscription );
	}

	/**
	 * When a subscription is switched, maybe schedule some events.
	 * 
	 * @param WC_Order $order
	 */
	public function maybe_schedule_when_switched( $order ) {
		$switch_order_data = wcs_get_objects_property( $order, 'subscription_switch_data' );

		if ( ! empty( $switch_order_data ) ) {
			foreach ( $switch_order_data as $subscription_id => $switch_data ) {
				$subscription = wcs_get_subscription( $subscription_id );

				if ( $subscription instanceof WC_Subscription ) {
					$this->maybe_schedule_when_status_updated( $subscription, $subscription->get_status() );
				}
			}
		}
	}

	/**
	 * Loop all subscriptions and schedule the reminders.
	 */
	public function maybe_schedule_reminders_for_existing_subscriptions() {
		$subscriptions = wcs_get_subscriptions( array(
			'subscription_status'    => array( 'active', 'expired', 'pending-cancel' ), // Include "pending-cancel" for BKWD compatibility
			'subscriptions_per_page' => -1,
			'return'                 => 'ids',
				) );

		if ( empty( $subscriptions ) ) {
			return;
		}

		foreach ( $subscriptions as $subscription_id => $subscription ) {
			// Apply product meta
			$this->apply_product_meta_for_old_subscription( $subscription_id );
			// Populate
			$subscription = wcs_get_subscription( $subscription_id );

			if ( $subscription ) {
				// Run the schedule
				$this->maybe_schedule_when_status_updated( $subscription, $subscription->get_status() );
			}
		}
	}

	/**
	 * Remind the subscriber before the trial is ending.
	 * 
	 * @param int $subscription_id The ID of a 'shop_subscription' post
	 * @param int $remind_day
	 */
	public function remind_trial_ending( $subscription_id, $remind_day ) {
		$subscription = wcs_get_subscription( $subscription_id );

		if ( $subscription && ! empty( $subscription->get_time( 'trial_end' ) ) ) {
			/**
			 * Remind trial ending.
			 * 
			 * @since version 1.0.0
			 */
			do_action( 'asp_erws_woocommerce_subscriptions_remind_trial_ending', $subscription, $remind_day );
		}
	}

	/**
	 * Remind the subscriber before the subscription due.
	 * 
	 * @param int $subscription_id The ID of a 'shop_subscription' post
	 * @param int $remind_day
	 */
	public function remind_renewal( $subscription_id, $remind_day ) {
		$subscription = wcs_get_subscription( $subscription_id );

		if ( $subscription ) {
			if ( $subscription->is_manual() ) {
				/**
				 * Remind manual renewal.
				 * 
				 * @since version 1.0.0
				 */
				do_action( 'asp_erws_woocommerce_subscriptions_remind_manual_renewal', $subscription, $remind_day );
			} else {
				/**
				 * Remind auto renewal.
				 * 
				 * @since version 1.0.0
				 */
				do_action( 'asp_erws_woocommerce_subscriptions_remind_auto_renewal', $subscription, $remind_day );
			}
		}
	}

	/**
	 * Remind the subscriber before the subscription gets expired.
	 * 
	 * @param int $subscription_id The ID of a 'shop_subscription' post
	 * @param int $remind_day
	 */
	public function remind_expiration( $subscription_id, $remind_day ) {
		$subscription = wcs_get_subscription( $subscription_id );

		if ( $subscription && ! empty( $subscription->get_time( 'end' ) ) ) {
			/**
			 * Remind expiration.
			 * 
			 * @since version 1.0.0
			 */
			do_action( 'asp_erws_woocommerce_subscriptions_remind_expiration', $subscription, $remind_day );
		}
	}

	/**
	 * Remind the subscriber after the subscription gets expired.
	 * 
	 * @param int $subscription_id The ID of a 'shop_subscription' post
	 * @param int $remind_day
	 */
	public function remind_after_expiry( $subscription_id, $remind_day ) {
		$subscription = wcs_get_subscription( $subscription_id );

		if ( $subscription && $subscription->has_status( 'expired' ) ) {
			/**
			 * Remind after expiry.
			 * 
			 * @since version 2.1.0
			 */
			do_action( 'asp_erws_woocommerce_subscriptions_remind_after_expiry', $subscription, $remind_day );
		}
	}
}
