<?php

defined( 'ABSPATH' ) || exit;

/**
 * Check whether the given the value is custom email reminder template. 
 *
 * @param  mixed $the_custom_template Post object or post ID of the custom email reminder template.
 * @return bool True on success.
 */
function asp_erws_is_custom_template( $the_custom_template ) {
	$is_template = false;

	if ( is_object( $the_custom_template ) && is_a( $the_custom_template, 'ASP_ERWS_Custom_Template' ) ) {
		$is_template = true;
	} elseif ( is_numeric( $the_custom_template ) && 'asp_erws_custom_tmpl' === get_post_type( $the_custom_template ) ) {
		$is_template = true;
	}

	return $is_template;
}

/**
 * Get the custom email reminder template.
 * 
 * @param ASP_ERWS_Custom_Template $custom_template
 * @param bool $wp_error
 * @return bool|\ASP_ERWS_Custom_Template
 */
function asp_erws_get_custom_template( $custom_template, $wp_error = false ) {
	if ( ! $custom_template ) {
		return false;
	}

	try {
		$custom_template = new ASP_ERWS_Custom_Template( $custom_template );
	} catch ( Exception $e ) {
		return $wp_error ? new WP_Error( 'error', $e->getMessage() ) : false;
	}

	return $custom_template;
}

/**
 * Get the custom email reminder template statuses.
 * 
 * @return array
 */
function asp_erws_get_custom_template_statuses() {
	$template_statuses = array(
		'asp-active'   => __( 'Active', 'email-reminders-for-woocommerce-subscriptions' ),
		'asp-inactive' => __( 'Inactive', 'email-reminders-for-woocommerce-subscriptions' ),
	);

	return $template_statuses;
}

/**
 * Trim our post status without prefix.
 * 
 * @param string $status
 * @return string
 */
function asp_erws_trim_post_status( $status ) {
	$status = 'asp-' === substr( $status, 0, 4 ) ? substr( $status, 4 ) : $status;
	return $status;
}

/**
 * Get the custom email reminder template status name.
 * 
 * @param string $status
 * @return string
 */
function asp_erws_get_custom_template_status_name( $status ) {
	$statuses = asp_erws_get_custom_template_statuses();
	$status   = asp_erws_trim_post_status( $status );
	$status   = isset( $statuses[ "asp-{$status}" ] ) ? $statuses[ "asp-{$status}" ] : $status;
	return $status;
}

/**
 * See if a string is an custom email reminder template status.
 *
 * @param  string $maybe_status Status, including any asp- prefix.
 * @return bool
 */
function asp_erws_is_custom_template_status( $maybe_status ) {
	$statuses = asp_erws_get_custom_template_statuses();
	return isset( $statuses[ $maybe_status ] );
}

/**
 * Prepare the custom email reminder template name to display.
 * 
 * @param ASP_ERWS_Custom_Template $custom_template
 * @return string 
 */
function asp_erws_get_custom_template_name( $custom_template ) {
	if ( asp_erws_is_custom_template( $custom_template ) ) {
		return sprintf( '%1$s (#%2$s)', $custom_template->get_name(), $custom_template->get_id() );
	} else {
		return '';
	}
}

/**
 * Return the array of custom email reminder templates based upon the args requested.
 * 
 * @param array $args
 * @return object
 */
function asp_erws_get_custom_templates( $args = array() ) {
	global $wpdb;
	$wpdb_ref = &$wpdb;

	$args = wp_parse_args( $args, array(
		'status'                 => array_keys( asp_erws_get_custom_template_statuses() ),
		'include_ids'            => array(),
		'exclude_ids'            => array(),
		'email_slug'             => '',
		'email_sending_interval' => '',
		's'                      => '',
		'page'                   => 1,
		'limit'                  => -1,
		'paginate'               => true,
		'return'                 => 'objects',
		'orderby'                => 'menu_order',
		'order'                  => 'ASC',
			) );

	// Add the 'asp-' prefix to status if needed.
	if ( ! empty( $args[ 'status' ] ) ) {
		if ( is_array( $args[ 'status' ] ) ) {
			foreach ( $args[ 'status' ] as &$status ) {
				if ( asp_erws_is_custom_template_status( 'asp-' . $status ) ) {
					$status = 'asp-' . $status;
				}
			}
		} elseif ( asp_erws_is_custom_template_status( 'asp-' . $args[ 'status' ] ) ) {
			$args[ 'status' ] = 'asp-' . $args[ 'status' ];
		}
	}

	// Statuses
	if ( is_array( $args[ 'status' ] ) ) {
		$allowed_statuses = " AND post_status IN ('" . implode( "','", $args[ 'status' ] ) . "') ";
	} else {
		$allowed_statuses = " AND post_status = '" . esc_sql( $args[ 'status' ] ) . "' ";
	}

	// Search term
	if ( ! empty( $args[ 's' ] ) ) {
		$term          = str_replace( '#', '', wc_clean( wp_unslash( $args[ 's' ] ) ) );
		$search_fields = array();

		$search_where = " AND ( 
                 (ID LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR
                 (post_title LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR
                 (post_excerpt LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR
                 (post_content LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%') OR 
                 (pm.meta_value LIKE '%%" . $wpdb_ref->esc_like( $term ) . "%%' AND pm.meta_key IN ('" . implode( "','", array_map( 'esc_sql', $search_fields ) ) . "'))
                ) ";
	} else {
		$search_where = '';
	}

	// Includes
	if ( ! empty( $args[ 'include_ids' ] ) ) {
		$include_ids = " AND ID IN ('" . implode( "','", $args[ 'include_ids' ] ) . "') ";
	} else {
		$include_ids = '';
	}

	// Excludes
	if ( ! empty( $args[ 'exclude_ids' ] ) ) {
		$exclude_ids = " AND ID NOT IN ('" . implode( "','", $args[ 'exclude_ids' ] ) . "') ";
	} else {
		$exclude_ids = '';
	}

	// Allow core meta
	if ( ! empty( $args[ 'email_slug' ] ) ) {
		$allowed_email_meta = " INNER JOIN {$wpdb_ref->postmeta} AS pm2 ON (p.ID = pm2.post_id AND pm2.meta_value = '" . esc_sql( $args[ 'email_slug' ] ) . "' AND pm2.meta_key='_email_slug' ) ";
	} else {
		$allowed_email_meta = '';
	}

	// Allow core meta
	if ( ! empty( $args[ 'email_sending_interval' ] ) ) {
		$allowed_email_interval_meta = " INNER JOIN {$wpdb_ref->postmeta} AS pm3 ON (p.ID = pm3.post_id AND pm3.meta_value = '" . esc_sql( $args[ 'email_sending_interval' ] ) . "' AND pm3.meta_key='_email_sending_interval' ) ";
	} else {
		$allowed_email_interval_meta = '';
	}

	// Order by
	switch ( ! empty( $args[ 'orderby' ] ) ? $args[ 'orderby' ] : 'menu_order' ) {
		case 'ID':
			$orderby = ' ORDER BY ' . esc_sql( $args[ 'orderby' ] ) . ' ';
			break;
		default:
			$orderby = ' ORDER BY menu_order ';
			break;
	}

	// Order
	if ( ! empty( $args[ 'order' ] ) && 'desc' === strtolower( $args[ 'order' ] ) ) {
		$order = ' DESC ';
	} else {
		$order = ' ASC ';
	}

	// Paging
	if ( $args[ 'limit' ] >= 0 ) {
		$page   = absint( $args[ 'page' ] );
		$page   = $page ? $page : 1;
		$offset = absint( ( $page - 1 ) * $args[ 'limit' ] );
		$limits = 'LIMIT ' . $offset . ', ' . $args[ 'limit' ];
	} else {
		$limits = '';
	}

	$custom_template_ids = $wpdb_ref->get_col(
			$wpdb_ref->prepare( "
                             SELECT DISTINCT ID FROM {$wpdb_ref->posts} AS p
                             INNER JOIN {$wpdb_ref->postmeta} AS pm ON (p.ID = pm.post_id)
                             $allowed_email_meta $allowed_email_interval_meta
                             WHERE 1=1 AND post_type = '%s' $allowed_statuses $search_where $include_ids $exclude_ids
                             $orderby $order $limits 
                            ", 'asp_erws_custom_tmpl'
			) );

	if ( 'objects' === $args[ 'return' ] ) {
		$custom_templates = array_filter( array_combine( $custom_template_ids, array_map( 'asp_erws_get_custom_template', $custom_template_ids ) ) );
	} else {
		$custom_templates = $custom_template_ids;
	}

	if ( $args[ 'paginate' ] ) {
		$custom_templates_count = count( $custom_templates );
		$return                 = ( object ) array(
					'templates'     => $custom_templates,
					'total'         => $custom_templates_count,
					'has_template'  => $custom_templates_count > 0,
					'max_num_pages' => $args[ 'limit' ] > 0 ? ceil( $custom_templates_count / $args[ 'limit' ] ) : 1,
		);
	} else {
		$return = $custom_templates;
	}

	/**
	 * Get custom email reminder templates.
	 * 
	 * @since 1.8.0
	 */
	return apply_filters( 'asp_erws_got_custom_templates', $return, $args );
}

/**
 * Get all terms for a product by ID, including hierarchy
 *
 * @param  int $product_id Product ID.
 * @param  string $taxonomy Taxonomy slug.
 * @return array
 */
function asp_erws_get_product_term_ids_deep( $product_id, $taxonomy = 'product_cat' ) {
	$product_term_ids = wc_get_product_term_ids( $product_id, $taxonomy );

	foreach ( $product_term_ids as $product_term ) {
		$product_term_ids = array_merge( $product_term_ids, get_ancestors( $product_term, $taxonomy ) );
	}

	return $product_term_ids;
}

/**
 * Gets the WPML translated object ID.
 * 
 * @param int $element_id
 * @param string $lang
 * @param string $element_type
 * @param bool $return_original_if_missing
 * @return int
 */
function asp_erws_maybe_get_wpml_translated_object_id( $element_id, $lang, $element_type = 'post', $return_original_if_missing = false ) {
	global $sitepress;

	if ( $sitepress && ! empty( $lang ) ) {
		return $sitepress->get_object_id( $element_id, $element_type, $return_original_if_missing, $lang );
	}

	return $element_id;
}

/**
 * Get the custom template ID for subscription.
 * 
 * @param WC_Subscription $subscription
 * @param string $email_slug
 * @param string $email_sending_interval
 * @return int|null
 */
function asp_erws_get_custom_template_for_subscription( $subscription, $email_slug, $email_sending_interval ) {
	if ( ! is_a( $subscription, 'WC_Subscription' ) || empty( $email_slug ) || empty( $email_sending_interval ) ) {
		return null;
	}

	$custom_templates = asp_erws_get_custom_templates( array(
		'status'                 => 'active',
		'email_slug'             => $email_slug,
		'email_sending_interval' => $email_sending_interval,
			) );

	if ( ! $custom_templates->has_template ) {
		return null;
	}

	$canonical_product_ids      = array();
	$canonical_product_term_ids = array();

	foreach ( $subscription->get_items( 'line_item' ) as $line_item ) {
		$canonical_product_ids[]    = $line_item[ 'variation_id' ] > 0 ? $line_item[ 'variation_id' ] : $line_item[ 'product_id' ];
		$canonical_product_term_ids = array_unique( array_merge( $canonical_product_term_ids, asp_erws_get_product_term_ids_deep( $line_item[ 'product_id' ] ) ) );
	}

	$user = get_user_by( 'id', $subscription->get_user_id() );

	foreach ( $custom_templates->templates as $custom_template ) {
		$is_product_valid = false;
		switch ( $custom_template->get_criteria_product_filter() ) {
			case 'included-products':
				if ( ! empty( $canonical_product_ids ) && count( array_intersect( $canonical_product_ids, $custom_template->get_criteria_products() ) ) > 0 ) {
					$is_product_valid = true;
				}
				break;
			case 'included-product-cats':
				if ( ! empty( $canonical_product_term_ids ) && count( array_intersect( $canonical_product_term_ids, $custom_template->get_criteria_product_cats() ) ) > 0 ) {
					$is_product_valid = true;
				}
				break;
			default:
				$is_product_valid = true;
				break;
		}

		$is_user_valid = false;
		switch ( $custom_template->get_criteria_user_filter() ) {
			case 'included-users':
				if ( ! empty( $user ) && in_array( $user->ID, $custom_template->get_criteria_users() ) ) {
					$is_user_valid = true;
				}
				break;
			case 'included-user-roles':
				if ( ! empty( $user ) && count( array_intersect( $user->roles, $custom_template->get_criteria_user_roles() ) ) > 0 ) {
					$is_user_valid = true;
				}
				break;
			default:
				$is_user_valid = true;
				break;
		}

		if ( $is_product_valid && $is_user_valid ) {
			return asp_erws_maybe_get_wpml_translated_object_id( $custom_template->get_id(), $subscription->get_meta( 'wpml_language' ) );
		}
	}

	return null;
}
