<?php
defined( 'ABSPATH' ) || exit;

/**
 * Email Reminders for WooCommerce Subscriptions Admin.
 * 
 * @class ASP_ERWS_Admin
 * @package Class
 */
class ASP_ERWS_Admin {

	/**
	 * Success messages.
	 *
	 * @var array
	 */
	protected static $success_messages = array();

	/**
	 * Error messages.
	 *
	 * @var array
	 */
	protected static $error_messages = array();

	/**
	 * Init ASP_ERWS_Admin.
	 */
	public static function init() {
		add_action( 'admin_enqueue_scripts', __CLASS__ . '::admin_enqueues', 11 );
		add_filter( 'woocommerce_screen_ids', __CLASS__ . '::wc_screen_ids' );
		add_filter( 'woocommerce_subscription_settings', __CLASS__ . '::subscription_settings', 99 );
		add_action( 'woocommerce_product_options_general_product_data', __CLASS__ . '::edit_subscription_product_fields' );
		add_action( 'woocommerce_variable_subscription_pricing', __CLASS__ . '::edit_subscription_variation_fields', 10, 3 );
		add_action( 'woocommerce_admin_field_asp_erws_existing_subscriptions_scheduler', __CLASS__ . '::add_existing_subscriptions_scheduler' );
		add_action( 'woocommerce_process_product_meta', __CLASS__ . '::save_subscription_product_meta' );
		add_action( 'woocommerce_save_product_variation', __CLASS__ . '::save_subscription_variation_meta', 10, 2 );
		add_action( 'woocommerce_process_product_meta', __CLASS__ . '::init_background_process', 20, 1 );

		ASP_ERWS_Admin_Post_Types::init();
	}

	/**
	 * Get our screen ids.
	 *
	 * @return array
	 */
	public static function get_screen_ids() {
		$wc_screen_id = sanitize_title( __( 'WooCommerce', 'woocommerce' ) );
		$screen_ids   = array(
			'asp_erws_custom_tmpl',
			'edit-asp_erws_custom_tmpl',
			$wc_screen_id . '_page_wc-settings',
		);

		/**
		 * Get admin screen ids.
		 * 
		 * @since 1.8.0
		 */
		return apply_filters( 'asp_erws_screen_ids', $screen_ids );
	}

	/**
	 * Return the array of categories for the products.
	 * 
	 * @param  array $args
	 * @return array
	 */
	public static function get_product_terms( $args = array() ) {
		$categories = array();
		$args       = wp_parse_args( $args, array( 'taxonomy' => 'product_cat', 'orderby' => 'name' ) );
		$terms      = get_terms( $args );

		if ( is_array( $terms ) ) {
			foreach ( $terms as $term ) {
				$categories[ $term->term_id ] = $term->name;
			}
		}

		return $categories;
	}

	/**
	 * Get WP User roles
	 * 
	 * @return array
	 */
	public static function get_user_roles() {
		global $wp_roles;

		$user_roles = array();
		foreach ( $wp_roles->roles as $user_role_key => $user_role ) {
			$user_roles[ $user_role_key ] = $user_role[ 'name' ];
		}

		$user_roles[ 'guest' ] = 'Guest';
		return $user_roles;
	}

	/**
	 * Add an success message.
	 *
	 * @param string $text Success to add.
	 */
	public static function add_success( $text ) {
		self::$success_messages[] = $text;
	}

	/**
	 * Add an error message.
	 *
	 * @param string $text Error to add.
	 */
	public static function add_error( $text ) {
		self::$error_messages[] = $text;
	}

	/**
	 * Display a notice message.
	 * 
	 * @param string $text Notice to display.
	 */
	public static function print_notice( $text ) {
		if ( ! empty( $text ) ) {
			echo '<div class="asp-erws-notice">';
			echo '<p>' . wp_kses_post( $text ) . '</p>';
			echo '</div>';
		}
	}

	/**
	 * Check the screen against the which context.
	 * 
	 * @param string $screen_id
	 * @param string $which
	 * @return bool
	 */
	public static function is_screen( $screen_id, $which = 'any' ) {
		if ( in_array( $screen_id, self::get_screen_ids(), true ) ) {
			if ( is_array( $which ) ) {
				return in_array( $screen_id, $which );
			} else if ( 'any' !== $which ) {
				return ( $screen_id === $which );
			} else {
				return true;
			}
		}

		return false;
	}

	/**
	 * Enqueue scripts and styles.
	 */
	public static function admin_enqueues() {
		global $post;
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';

		if ( in_array( $screen_id, self::get_screen_ids(), true ) || 'product' === $screen_id ) {
			wp_register_script( 'asp-erws-admin', ASP_ERWS()->plugin_url() . '/assets/js/admin.js', array( 'jquery', 'wc-enhanced-select', 'wc-admin-meta-boxes', 'editor', 'quicktags' ), ASP_ERWS_VERSION );
			wp_register_style( 'asp-erws-admin', ASP_ERWS()->plugin_url() . '/assets/css/admin.css', array( 'woocommerce_admin_styles', 'wc-admin-layout' ), ASP_ERWS_VERSION, 'all', false );

			wp_localize_script( 'asp-erws-admin', 'asp_erws_admin_params', array(
				'post_id'                                => isset( $post->ID ) ? $post->ID : 0,
				'ajax_url'                               => admin_url( 'admin-ajax.php' ),
				'existing_subscriptions_scheduler_nonce' => wp_create_nonce( 'asp-erws-existing-subscriptions-scheduler' ),
				'custom_template_email_data_nonce'       => wp_create_nonce( 'asp-erws-custom-template-email-data' ),
				'custom_template_email_test_nonce'       => wp_create_nonce( 'asp-erws-test-custom-template-email' ),
				'i18n_scheduled_success'                 => esc_attr__( 'The scheduling process is successfully initiated and it runs in the background. Each reminder will be scheduled on a queue basis which may take a little while, so please be patient.', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_scheduled_error'                   => esc_attr__( 'Please enter atleast any one of the reminder values to schedule !!', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_prompt_schedule_now'               => esc_attr__( 'Are you sure you want to apply the reminder settings for your existing subscriptions?', 'email-reminders-for-woocommerce-subscriptions' ),
				'back_to_settings_url'                   => esc_url( admin_url( 'admin.php?page=wc-settings&tab=subscriptions#erws_reminders-description' ) ),
				'back_to_all_custom_templates_url'       => esc_url( admin_url( 'edit.php?post_type=asp_erws_custom_tmpl' ) ),
				'i18n_back_to_all_label'                 => esc_attr__( 'Back', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_back_to_settings_label'            => esc_attr__( 'Back to settings', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_custom_template_description'       => esc_attr__( 'Template content', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_custom_template_creation_warning'  => esc_attr__( 'Please configure reminder intervals under WooCommerce > Settings > Subscriptions > Email Reminders before creating the custom reminder template !', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_email_sent'                        => esc_attr__( 'Email sent!', 'email-reminders-for-woocommerce-subscriptions' ),
				'i18n_email_failed'                      => esc_attr__( 'Failed to send email!', 'email-reminders-for-woocommerce-subscriptions' ),
			) );

			wp_enqueue_script( 'asp-erws-admin' );
			wp_enqueue_style( 'asp-erws-admin' );
			wp_enqueue_media();

			if ( self::is_screen( $screen_id, array( 'edit-asp_erws_custom_tmpl' ) ) ) {
				wp_enqueue_script( 'asp-erws-post-ordering', ASP_ERWS()->plugin_url() . '/assets/js/post-ordering.js', array( 'jquery-ui-sortable' ), ASP_ERWS_VERSION );
			}

			if ( self::is_screen( $screen_id, array( 'asp_erws_custom_tmpl', 'edit-asp_erws_custom_tmpl' ) ) ) {
				wp_dequeue_script( 'autosave' );
			}
		}
	}

	/**
	 * Register our screens to WC screens.
	 *
	 * @param array $screen_ids
	 * @return array
	 */
	public static function wc_screen_ids( $screen_ids ) {
		$screen = get_current_screen();

		if ( $screen && self::is_screen( $screen->id, 'asp_erws_custom_tmpl' ) ) {
			$screen_ids[] = $screen->id;
		}

		return $screen_ids;
	}

	/**
	 * Return the array of our settings to WC Subscriptions.
	 * 
	 * @param array $settings
	 * @return array
	 */
	public static function subscription_settings( $settings ) {
		$misc_section_start = wp_list_filter( $settings, array( 'type' => 'title', 'id' => 'woocommerce_subscriptions_miscellaneous' ) );

		$reminders_settings   = array();
		$reminders_settings[] = array(
			'name' => _x( 'Email Reminders', 'options section heading', 'email-reminders-for-woocommerce-subscriptions' ),
			'type' => 'title',
			'id'   => 'erws_reminders',
			'desc' => sprintf(
					/* translators: 1: template create/edit link */
					__(
							'<p>
                                This section allows you to send different subscription email reminders to your customers at specified intervals.
                            </p>
                            <p>
                            &rarr; You can optionally %s to send different email content for various reminder intervals configured here for your customers. <br>&rarr; By default, emails from <b>WooCommerce > Settings > Emails</b> will be used if no custom templates are created for the respective reminder.
                            </p>'
							, 'email-reminders-for-woocommerce-subscriptions'
					),
					sprintf(
							'<a href="%1$s" class="button-small" target="_blank"><b>%2$s</b></a>',
							admin_url( 'edit.php?post_type=asp_erws_custom_tmpl' ),
							__( 'Create or Edit Custom Email Reminder Template', 'email-reminders-for-woocommerce-subscriptions' )
					)
			),
		);

		if ( has_filter( 'asp_erws_before_reminders_settings_array' ) ) {
			/**
			 * Before reminders settings array.
			 * 
			 * @since 2.3.1
			 */
			$reminders_settings[] = apply_filters( 'asp_erws_before_reminders_settings_array', array() );
		}

		$reminders_settings[] = array(
			'name'        => __( 'Remind Trial Ending', 'email-reminders-for-woocommerce-subscriptions' ),
			'id'          => 'erws_trial_ending_reminder_days',
			'type'        => 'text',
			'default'     => '',
			'placeholder' => __( 'e.g. 3,2,1', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc'        => __( 'day(s) before trial end date.', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc_tip'    => __( 'Specify the day(s) (e.g., 3, 2, 1) before the trial end date to notify customers, in descending order.', 'email-reminders-for-woocommerce-subscriptions' ),
		);
		$reminders_settings[] = array(
			'name'        => __( 'Remind Auto Renewal', 'email-reminders-for-woocommerce-subscriptions' ),
			'id'          => 'erws_auto_renewal_reminder_days',
			'type'        => 'text',
			'default'     => '',
			'placeholder' => __( 'e.g. 3,2,1', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc'        => __( 'day(s) before due date.', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc_tip'    => __( 'Specify the day(s) (e.g., 3, 2, 1) before the subscription due date to notify customers for automatic renewals, in descending order.', 'email-reminders-for-woocommerce-subscriptions' ),
		);
		$reminders_settings[] = array(
			'name'        => __( 'Remind Manual Renewal', 'email-reminders-for-woocommerce-subscriptions' ),
			'id'          => 'erws_manual_renewal_reminder_days',
			'type'        => 'text',
			'default'     => '',
			'placeholder' => __( 'e.g. 3,2,1', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc'        => __( 'day(s) before due date.', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc_tip'    => __( 'Specify the day(s) (e.g., 3, 2, 1) before the subscription due date to notify customers for manual renewals, in descending order.', 'email-reminders-for-woocommerce-subscriptions' ),
		);
		$reminders_settings[] = array(
			'name'        => __( 'Remind Expiration', 'email-reminders-for-woocommerce-subscriptions' ),
			'id'          => 'erws_expiration_reminder_days',
			'type'        => 'text',
			'default'     => '',
			'placeholder' => __( 'e.g. 3,2,1', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc'        => __( 'day(s) before expiry date.', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc_tip'    => __( 'Specify the day(s) (e.g., 3, 2, 1) before the subscription expiry date to notify customers, in descending order.', 'email-reminders-for-woocommerce-subscriptions' ),
		);
		$reminders_settings[] = array(
			'name'        => __( 'Remind After Expiry', 'email-reminders-for-woocommerce-subscriptions' ),
			'id'          => 'erws_after_expiry_reminder_days',
			'type'        => 'text',
			'default'     => '',
			'placeholder' => __( 'e.g. 1,2,3', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc'        => __( 'day(s) after expiry date.', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc_tip'    => __( 'Specify the day(s) (e.g., 1, 2, 3) after the subscription expiry date to notify customers, in ascending order.', 'email-reminders-for-woocommerce-subscriptions' ),
		);

		if ( has_filter( 'asp_erws_after_reminders_settings_array' ) ) {
			/**
			 * After reminders settings array.
			 * 
			 * @since 2.3.1
			 */
			$reminders_settings[] = apply_filters( 'asp_erws_after_reminders_settings_array', array() );
		}

		$reminders_settings[] = array( 'type' => 'asp_erws_existing_subscriptions_scheduler' );
		$reminders_settings[] = array( 'type' => 'sectionend', 'id' => 'erws_reminders' );

		array_splice( $settings, key( $misc_section_start ), 0, $reminders_settings );

		return $settings;
	}

	/**
	 * Subscription fields on product edit page.
	 */
	public static function edit_subscription_product_fields() {
		global $post;
		$product_object          = wc_get_product( $post->ID );
		$is_variable             = $product_object && $product_object->is_type( 'variable' );
		$exclude_reminder_emails = $is_variable ? '' : get_post_meta( $post->ID, '_asp_erws_exclude_reminder_emails', true );
		?>
		<div class="options_group show_if_subscription hidden asp_erws_subscription_fields_wrapper">
			<p class="form-field asp_erws_exclude_reminder_emails_field">
				<label for="asp_erws_exclude_reminder_emails"><?php esc_html_e( 'Exclude reminder emails', 'email-reminders-for-woocommerce-subscriptions' ); ?></label>
				<select class="wc-enhanced-select" name="asp_erws_exclude_reminder_emails[]" id="asp_erws_exclude_reminder_emails" multiple="multiple">
					<?php foreach ( ASP_ERWS()->get_reminders() as $slug => $val ) { ?>
						<option value="<?php echo esc_attr( $slug ); ?>"
						<?php
						if ( is_array( $exclude_reminder_emails ) ) {
							selected( in_array( ( string ) $slug, $exclude_reminder_emails, true ), true );
						} else {
							selected( $exclude_reminder_emails, ( string ) $slug );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
					<?php } ?>
				</select>
				<span class="woocommerce-help-tip" data-tip="<?php esc_attr_e( 'Reminder emails which are selected here will not be sent to the customers.', 'email-reminders-for-woocommerce-subscriptions' ); ?>"></span>
			</p>
		</div>
		<?php
	}

	/**
	 * Subscription fields on product edit page for variation.
	 */
	public static function edit_subscription_variation_fields( $loop, $variation_data, $variation ) {
		$exclude_reminder_emails = get_post_meta( $variation->ID, '_asp_erws_exclude_reminder_emails', true );
		?>
		<div class="show_if_variable-subscription asp_erws_subscription_fields_wrapper">
			<p class="form-row form-row-first form-field show_if_variable-subscription asp_erws_exclude_reminder_emails_field">
				<label for="asp_erws_exclude_reminder_emails[<?php echo esc_attr( $loop ); ?>]"><?php esc_html_e( 'Exclude reminder emails', 'email-reminders-for-woocommerce-subscriptions' ); ?>
					<span class="woocommerce-help-tip" data-tip="<?php esc_attr_e( 'Reminder emails which are selected here will not be sent to the customers.', 'email-reminders-for-woocommerce-subscriptions' ); ?>"></span>
				</label>
				<select class="wc-enhanced-select" name="asp_erws_exclude_reminder_emails[<?php echo esc_attr( $loop ); ?>][]" multiple="multiple">
					<?php foreach ( ASP_ERWS()->get_reminders() as $slug => $val ) { ?>
						<option value="<?php echo esc_attr( $slug ); ?>"
						<?php
						if ( is_array( $exclude_reminder_emails ) ) {
							selected( in_array( ( string ) $slug, $exclude_reminder_emails, true ), true );
						} else {
							selected( $exclude_reminder_emails, ( string ) $slug );
						}
						?>
								>
									<?php echo esc_html( $val ); ?>
						</option>
					<?php } ?>
				</select>
			</p>
		</div>
		<?php
	}

	/**
	 * Adds the existing subscriptions scheduler HTML.
	 */
	public static function add_existing_subscriptions_scheduler() {
		$tooltip_html = wc_help_tip( __( 'If you want to apply reminder settings for your existing subscriptions then click the "Schedule Now" button to schedule the reminders.', 'email-reminders-for-woocommerce-subscriptions' ) );
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="existing_subscriptions_scheduler"><?php esc_html_e( 'Reminders for Existing Subscriptions', 'email-reminders-for-woocommerce-subscriptions' ); ?> <?php echo wp_kses_post( $tooltip_html ); ?></label>
			</th>
			<td class="forminp forminp-existing_subscriptions_scheduler">
				<button class="button-primary woocommerce-save-button components-button is-primary" id="asp_erws_existing_subscriptions_scheduler">
					<?php esc_html_e( 'Schedule now', 'email-reminders-for-woocommerce-subscriptions' ); ?>
				</button>
			</td>
		</tr>
		<?php
	}

	/**
	 * Save subscription product meta.
	 */
	public static function save_subscription_product_meta( $product_id ) {
		$posted       = asp_erws_get_global_var( 'post' );
		$product_type = empty( $posted[ 'product-type' ] ) ? WC_Product_Factory::get_product_type( $product_id ) : sanitize_title( wp_unslash( $posted[ 'product-type' ] ) );
		$product_type = $product_type ? $product_type : 'simple';

		if ( ! empty( $posted[ '_wcsnonce' ] ) && wp_verify_nonce( sanitize_key( wp_unslash( $posted[ '_wcsnonce' ] ) ), 'wcs_subscription_meta' ) && 'subscription' === $product_type ) {
			if ( isset( $posted[ 'asp_erws_exclude_reminder_emails' ] ) ) {
				update_post_meta( $product_id, '_asp_erws_exclude_reminder_emails', ! is_array( $posted[ 'asp_erws_exclude_reminder_emails' ] ) ? array_filter( explode( ',', wc_clean( wp_unslash( $posted[ 'asp_erws_exclude_reminder_emails' ] ) ) ) ) : wc_clean( wp_unslash( $posted[ 'asp_erws_exclude_reminder_emails' ] ) )  );
			} else {
				update_post_meta( $product_id, '_asp_erws_exclude_reminder_emails', array() );
			}
		}
	}

	/**
	 * Save subscription variation meta.
	 */
	public static function save_subscription_variation_meta( $variation_id, $loop ) {
		$posted       = asp_erws_get_global_var( 'post' );
		$product_type = empty( $posted[ 'product-type' ] ) ? WC_Product_Factory::get_product_type( $variation_id ) : sanitize_title( wp_unslash( $posted[ 'product-type' ] ) );
		$product_type = $product_type ? $product_type : 'variable';

		if ( 'variable-subscription' === $product_type && ! empty( $posted[ '_wcsnonce_save_variations' ] ) && wp_verify_nonce( sanitize_key( wp_unslash( $posted[ '_wcsnonce_save_variations' ] ) ), 'wcs_subscription_variations' ) ) {
			if ( isset( $posted[ 'asp_erws_exclude_reminder_emails' ][ $loop ] ) ) {
				update_post_meta( $variation_id, '_asp_erws_exclude_reminder_emails', ! is_array( $posted[ 'asp_erws_exclude_reminder_emails' ][ $loop ] ) ? array_filter( explode( ',', wc_clean( wp_unslash( $posted[ 'asp_erws_exclude_reminder_emails' ][ $loop ] ) ) ) ) : wc_clean( wp_unslash( $posted[ 'asp_erws_exclude_reminder_emails' ][ $loop ] ) )  );
			} else {
				update_post_meta( $variation_id, '_asp_erws_exclude_reminder_emails', array() );
			}
		}
	}

	/**
	 * Init the background process.
	 */
	public static function init_background_process( $post_id ) {
		if ( ! empty( $post_id ) && false === as_next_scheduled_action( 'asp_erws_scheduled_existing_subscriptions_scheduler' ) ) {
			as_schedule_single_action( time() + 1, 'asp_erws_scheduled_existing_subscriptions_scheduler' );
		}
	}
}

ASP_ERWS_Admin::init();
