<?php
/**
 * URL utility class.
 *
 * @package   EDD\SoftwareLicensing\Utils
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Utils;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * URL utility class.
 *
 * @since 3.9.0
 */
class URL {

	/**
	 * Get the URL from the user agent if it's not provided.
	 *
	 * @since 3.8.13
	 * @since 3.9.0 The method was moved to the URL class.
	 * @return string
	 */
	public static function get_url_from_user_agent() {
		if ( ! isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
			return '';
		}

		// Attempt to grab the URL from the user agent if no URL is specified.
		$domain = array_map( 'trim', explode( ';', $_SERVER['HTTP_USER_AGENT'] ) );

		return ! empty( $domain[1] ) ? trim( $domain[1] ) : '';
	}

	/**
	 * Check if the URL is local.
	 *
	 * @since 3.9.0
	 * @param string $url  The URL to check.
	 * @param string $host The host to check.
	 * @return bool
	 */
	public function is_local_url( string $url, string $host ): bool {
		$domain_patterns = new DomainPatterns();
		$patterns        = $domain_patterns->get_patterns();

		if ( $this->is_tld_local( $patterns, $host ) ) {
			return true;
		}

		if ( $this->is_subdomain_local( $patterns, $host ) ) {
			return true;
		}

		if ( $this->is_pattern_local( $patterns, $url ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if the TLD is local.
	 *
	 * @since 3.9.0
	 * @param array  $patterns The patterns.
	 * @param string $host The host.
	 * @return bool
	 */
	private function is_tld_local( array $patterns, string $host ): bool {
		$check_tlds = apply_filters( 'edd_sl_validate_tlds', true );
		if ( ! $check_tlds ) {
			return false;
		}

		$tlds_to_check = apply_filters(
			'edd_sl_url_tlds',
			$patterns['tlds']
		);

		foreach ( $tlds_to_check as $tld ) {
			if ( false !== strpos( $host, $tld ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if the subdomain is local.
	 *
	 * @since 3.9.0
	 * @param array  $patterns The patterns.
	 * @param string $host The host.
	 * @return bool
	 */
	private function is_subdomain_local( array $patterns, string $host ): bool {
		if ( substr_count( $host, '.' ) <= 1 ) {
			return false;
		}

		$subdomains_to_check = apply_filters(
			'edd_sl_url_subdomains',
			$patterns['subdomains']
		);

		foreach ( $subdomains_to_check as $subdomain ) {

			$subdomain = str_replace( '.', '(.)', $subdomain );
			$subdomain = str_replace( array( '*', '(.)' ), '(.*)', $subdomain );

			if ( preg_match( '/^(' . $subdomain . ')/', $host ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if the pattern is local.
	 *
	 * @since 3.9.0
	 * @param array  $patterns The patterns.
	 * @param string $url The URL.
	 * @return bool
	 */
	private function is_pattern_local( array $patterns, string $url ): bool {
		/**
		 * Filter the patterns used to determine if a URL is local or not.
		 *
		 * @since 3.8.11
		 * @param array $patterns An array of patterns to check against.
		 * @return array $patterns The array of patterns to check against.
		 */
		$patterns = apply_filters( 'edd_sl_local_url_patterns', $patterns );

		foreach ( $patterns['patterns'] as $pattern ) {
			if ( $this->is_valid_regex_pattern( $pattern ) && preg_match( $pattern, $url ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if a regex pattern is valid.
	 *
	 * @since 3.9.0
	 * @param string $pattern The regex pattern to validate.
	 * @return bool True if the pattern is valid, false otherwise.
	 */
	private function is_valid_regex_pattern( string $pattern ): bool {
		if ( empty( $pattern ) ) {
			return false;
		}

		return false !== @preg_match( $pattern, '' ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
	}
}
