<?php
/**
 * Remote Requests Log.
 *
 * @package     EDD\SoftwareLicensing\RemoteRequests
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Utils;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Logs class.
 *
 * @since 3.9.0
 */
class LogRequests {

	/**
	 * Logs a remote request for a license activation.
	 *
	 * @since 3.9.0
	 * @param int    $license_id The license ID.
	 * @param string $site       The URL making the request.
	 * @param array  $data       The request data.
	 * @param bool   $create_new Whether to create a new activation if one doesn't exist.
	 * @return int|bool
	 */
	public static function log( $license_id, $site, $data = array(), $create_new = true ) {
		if ( ! is_numeric( $license_id ) ) {
			$license_id = edd_software_licensing()->get_license_by_key( $license_id );
		}
		if ( empty( $license_id ) || empty( $site ) ) {
			return;
		}

		$activation_id = self::get_activation_id( $license_id, $site, $create_new );
		if ( ! empty( $activation_id ) && ! empty( $data ) ) {
			self::record_activation_data( $activation_id, $data );
		}

		return $activation_id;
	}

	/**
	 * Gets the activation ID for a license and site.
	 *
	 * @since 3.9.0
	 * @param int    $license_id The license ID.
	 * @param string $site       The site URL.
	 * @param bool   $create_new Whether to create a new activation if one doesn't exist.
	 * @return int|bool
	 */
	private static function get_activation_id( $license_id, $site, $create_new = true ) {
		$site   = untrailingslashit( edd_software_licensing()->clean_site_url( $site ) );
		$args   = array(
			'site_name'  => $site,
			'license_id' => $license_id,
		);
		$query  = new \EDD\SoftwareLicensing\Database\Queries\LicenseActivation();
		$exists = $query->query( $args );

		if ( empty( $exists ) ) {
			if ( ! $create_new ) {
				return false;
			}

			return $query->add_item(
				array(
					'license_id' => $license_id,
					'site_name'  => $site,
				)
			);
		}

		$activation = reset( $exists );
		// If the last_request is older than 2 minutes, update the last_request field.
		if ( empty( $activation->last_request ) || strtotime( $activation->last_request ) < strtotime( '-2 minutes' ) ) {
			// Nothing is being modified, but we need to update the last_request field.
			$query->update_item(
				$activation->id,
				array(
					'last_request' => current_time( 'mysql', true ),
				)
			);
		}

		return $activation->id;
	}

	/**
	 * Records additional data for an activation.
	 *
	 * @since 3.9.0
	 * @param int   $activation_id The activation ID.
	 * @param array $data          The data to store.
	 */
	private static function record_activation_data( $activation_id, $data ) {
		if ( empty( $data['item_id'] ) ) {
			return;
		}
		$data_id    = false;
		$product_id = absint( $data['item_id'] );
		$query      = new \EDD\SoftwareLicensing\Database\Queries\LicenseActivationsData();
		$args       = array(
			'activation_id' => $activation_id,
			'product_id'    => $product_id,
		);
		$exists     = $query->query( $args );
		if ( ! empty( $data['version'] ) ) {
			$args['product_version'] = sanitize_text_field( $data['version'] );
		}
		if ( empty( $exists ) ) {
			$data_id = $query->add_item( $args );
		} else {
			$activation_data = reset( $exists );
			$data_id         = $activation_data->id;
			$query->update_item(
				$data_id,
				$args
			);
		}
		if ( ! $data_id ) {
			edd_debug_log( 'Failed to record activation data for activation ID ' . $activation_id );
			return;
		}

		self::maybe_log_meta( $activation_id, $data );
	}

	/**
	 * Logs meta data for activation data.
	 *
	 * @since 3.9.0
	 * @param int   $activation_id The data ID.
	 * @param array $data    The data to store.
	 */
	private static function maybe_log_meta( $activation_id, $data = array() ) {
		if ( empty( $data ) ) {
			return;
		}
		self::maybe_add_product_id_to_activation_meta( $activation_id, $data );

		if ( empty( $data['allow_tracking'] ) ) {
			return;
		}

		// Only log meta data that is in the list of meta keys we want to log.
		$meta_to_log = array_intersect_key( $data, array_flip( array_keys( eddsl_get_activation_data_meta_keys() ) ) );
		if ( empty( $meta_to_log ) ) {
			return;
		}

		foreach ( $meta_to_log as $key => $value ) {
			update_metadata( 'edd_license_activation', $activation_id, $key, $value );
		}
	}

	/**
	 * Adds the product ID to the activation meta table.
	 * This is used to track which products are being activated, even with All Access.
	 *
	 * @since 3.9.0
	 * @param int   $activation_id The activation ID.
	 * @param array $data          The data to store.
	 */
	private static function maybe_add_product_id_to_activation_meta( $activation_id, $data ) {
		if ( empty( $data['item_id'] ) ) {
			return;
		}

		$product_id = absint( $data['item_id'] );

		// Get existing product IDs for this activation.
		$existing_product_ids = get_metadata( 'edd_license_activation', $activation_id, 'product_id', false );
		$existing_product_ids = array_map( 'absint', $existing_product_ids );

		// Check if this product ID already exists to avoid duplicates.
		if ( in_array( $product_id, $existing_product_ids, true ) ) {
			return;
		}

		return add_metadata( 'edd_license_activation', $activation_id, 'product_id', $product_id );
	}
}
