<?php
/**
 * Domain Patterns
 *
 * @package EDD\SoftwareLicensing\Utils
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Utils;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\ExtensionUtils\v2\Services\Request;

/**
 * Domain Patterns
 *
 * @since 3.9.0
 */
class DomainPatterns {

	/**
	 * The endpoint to request.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	private $endpoint = 'local-domain-patterns';

	/**
	 * Get the patterns.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public function get_patterns(): array {
		return wp_parse_args( $this->maybe_update_patterns(), $this->default_patterns() );
	}

	/**
	 * Gets the patterns from the setting, or updates them if they are expired.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function maybe_update_patterns(): array {
		if ( edd_is_doing_unit_tests() ) {
			return array();
		}

		$domain_patterns = get_option( 'edd_sl_domain_patterns', array() );
		if ( ! empty( $domain_patterns['data'] ) && $domain_patterns['expires'] > time() ) {
			return json_decode( $domain_patterns['data'], true );
		}

		$request = new Request( $this->endpoint );
		$request->make_request();
		if ( $request->get_response()->success ) {
			$domain_patterns['data']    = $request->get_response()->data;
			$domain_patterns['expires'] = time() + WEEK_IN_SECONDS;
			update_option( 'edd_sl_domain_patterns', $domain_patterns );

			return json_decode( $domain_patterns['data'], true );
		}

		return array();
	}

	/**
	 * Get the default patterns.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function default_patterns(): array {
		return array(
			'tlds'       => array(
				'.dev',
				'.local',
				'.test',
			),
			'subdomains' => array(
				'dev.',
				'*.staging.',
				'*.test.',
				'staging-*.',
				'*.wpengine.com',
				'*.instawp.xyz',
				'*.cloudwaysapps.com',
				'*.flywheelsites.com',
				'*.flywheelstaging.com',
				'*.myftpupload.com',
				'*.kinsta.cloud',
			),
			'patterns'   => array(
				'newfold' => '/\/staging\/\d{3,}/',
			),
		);
	}
}
