<?php
/**
 * Requirements Check
 *
 * @package EDD\SoftwareLicensing\Requirements
 * @copyright Sandhills Development, LLC
 * @license https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8
 */

namespace EDD\SoftwareLicensing\Requirements;

/**
 * Class Check
 *
 * @since 3.8
 */
final class Check {

	/**
	 * Plugin file
	 *
	 * @var string
	 * @since 3.8
	 */
	private $file;

	/**
	 * Plugin basename
	 *
	 * @var string
	 * @since 3.8
	 */
	private $plugin_base;

	/**
	 * Platform versions required to load Software Licensing.
	 *
	 * @var array[]
	 * @since 3.8
	 */
	private $current_requirements = array(
		'php'                    => array(
			'minimum' => '7.4',
			'name'    => 'PHP',
			'local'   => true,
		),
		'wp'                     => array(
			'minimum' => '6.2',
			'name'    => 'WordPress',
			'local'   => true,
		),
		'easy-digital-downloads' => array(
			'minimum' => '3.3.6',
			'name'    => 'Easy Digital Downloads',
			'local'   => true,
		),
	);

	/**
	 * The requirements object.
	 *
	 * @var Requirements
	 */
	private $requirements;

	/**
	 * Check constructor.
	 *
	 * @param string $plugin_file The path to the plugin file.
	 */
	public function __construct( $plugin_file ) {
		$this->file         = $plugin_file;
		$this->plugin_base  = plugin_basename( $this->file );
		$this->requirements = new Requirements( $this->current_requirements );
	}

	/**
	 * Loads the plugin if requirements have been met, otherwise
	 * displays "plugin not fully active" UI and exists.
	 *
	 * @since 3.8
	 */
	public function maybe_load() {
		$this->requirements->met() ? $this->load() : $this->quit();
	}

	/**
	 * Loads Software Licensing
	 *
	 * @since 3.8
	 */
	private function load() {
		require_once EDD_SL_PLUGIN_DIR . 'includes/misc-functions.php';
		if ( ! class_exists( 'EDD_Software_Licensing' ) ) {
			require_once EDD_SL_PLUGIN_DIR . 'includes/classes/class-edd-software-licensing.php';
		}

		$this->maybe_install();

		// Get Software Licensing running.
		edd_software_licensing();
	}

	/**
	 * Installs Software Licensing if needed.
	 *
	 * @since 3.8
	 */
	private function maybe_install() {
		if ( ! function_exists( 'edd_sl_install' ) ) {
			require_once EDD_SL_PLUGIN_DIR . 'includes/install.php';
		}

		if ( get_option( 'edd_sl_run_install' ) ) {
			// Install Software Licensing.
			edd_sl_install();

			// Delete this option so we don't run the install again.
			delete_option( 'edd_sl_run_install' );
		}
	}

	/**
	 * Adds action hooks to inject our "plugin not fully loaded" UI / text.
	 *
	 * @since 3.8
	 */
	private function quit() {
		add_action( 'admin_head', array( $this, 'admin_head' ) );
		add_action( "after_plugin_row_{$this->plugin_base}", array( $this, 'plugin_row_notice' ) );
	}

	/**
	 * Adds CSS to the admin head for our "plugin not fully loaded" UI.
	 *
	 * @since 3.8
	 */
	public function admin_head() {
		$name = $this->unmet_requirements_name();
		?>
		<style id="<?php echo esc_attr( $name ); ?>">
			.plugins tr[data-plugin="<?php echo esc_html( $this->plugin_base ); ?>"] th,
			.plugins tr[data-plugin="<?php echo esc_html( $this->plugin_base ); ?>"] td,
			.plugins .<?php echo esc_html( $name ); ?>-row th,
			.plugins .<?php echo esc_html( $name ); ?>-row td {
				background: #fff5f5;
			}

			.plugins tr[data-plugin="<?php echo esc_html( $this->plugin_base ); ?>"] th {
				box-shadow: none;
			}

			.plugins .<?php echo esc_html( $name ); ?>-row th span {
				margin-left: 6px;
				color: #dc3232;
			}

			.plugins tr[data-plugin="<?php echo esc_html( $this->plugin_base ); ?>"] th,
			.plugins .<?php echo esc_html( $name ); ?>-row th.check-column {
				border-left: 4px solid #dc3232 !important;
			}

			.plugins .<?php echo esc_html( $name ); ?>-row .column-description p {
				margin: 0;
				padding: 0;
			}

			.plugins .<?php echo esc_html( $name ); ?>-row .column-description p:not(:last-of-type) {
				margin-bottom: 8px;
			}
		</style>
		<?php
	}

	/**
	 * Displays a notice on the plugin row about missing requirements.
	 *
	 * @since 3.8
	 */
	public function plugin_row_notice() {
		$colspan = function_exists( 'wp_is_auto_update_enabled_for_type' ) && wp_is_auto_update_enabled_for_type( 'plugin' ) ? 2 : 1;
		?>
		<tr class="active <?php echo esc_attr( $this->unmet_requirements_name() ); ?>-row">
			<th class="check-column">
				<span class="dashicons dashicons-warning"></span>
			</th>
			<td class="column-primary">
				<?php $this->unmet_requirements_text(); ?>
			</td>
			<td class="column-description" colspan="<?php echo esc_attr( $colspan ); ?>">
				<?php $this->unmet_requirements_description(); ?>
			</td>
		</tr>
		<?php
	}

	/**
	 * Plugin specific text used in CSS to identify attribute IDs and classes.
	 *
	 * @since 3.8
	 * @return string
	 */
	private function unmet_requirements_name() {
		return 'edd-sl-requirements';
	}

	/**
	 * Outputs a message about the plugin not being fully active.
	 *
	 * @since 3.8
	 */
	private function unmet_requirements_text() {
		esc_html_e( 'This plugin is not fully active.', 'edd_sl' );
	}

	/**
	 * Displays error messages for all unmet requirements.
	 *
	 * @since 3.8
	 */
	private function unmet_requirements_description() {
		foreach ( $this->requirements->get_errors()->get_error_messages() as $message ) {
			echo wpautop( wp_kses_post( $message ) );
		}
	}
}
