<?php
/**
 * Request class.
 *
 * @package   EDD\SoftwareLicensing\RemoteRequests\Requests
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\RemoteRequests\Requests;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Request class.
 */
abstract class Request {

	/**
	 * Handle the request.
	 *
	 * @since 3.9.0
	 * @param array $data The data to handle.
	 */
	abstract public function handle( $data );

	/**
	 * Given an array of arguments, sort them by length, and then md5 them to generate a checksum.
	 *
	 * @since 3.5
	 * @param array $args The arguments to sort.
	 *
	 * @return string
	 */
	protected function get_request_checksum( $args = array() ) {
		usort( $args, array( $this, 'sort_args_by_length' ) );
		$string_args = wp_json_encode( $args );

		return md5( $string_args );
	}

	/**
	 * Send the API response data as a JSON response, and define the JSON_REQUEST and WP_REDIS_DISABLE_COMMENT constants.
	 *
	 * @since 3.6.12
	 * @param array $response_data The data to send to the api.
	 */
	protected function send_response( $response_data = array() ) {
		if ( ! defined( 'JSON_REQUEST' ) ) {
			define( 'JSON_REQUEST', true );
		}

		if ( ! defined( 'WP_REDIS_DISABLE_COMMENT' ) ) {
			define( 'WP_REDIS_DISABLE_COMMENT', true );
		}

		wp_send_json( $response_data );
	}

	/**
	 * Logs a remote request for a license activation.
	 *
	 * @since 3.9.0
	 * @param int    $license_id The license ID.
	 * @param string $site       The URL making the request.
	 * @param array  $data       The request data.
	 * @param bool   $create_new Whether to create a new activation if one doesn't exist.
	 * @return int|bool
	 */
	protected function log_remote_request( $license_id, $site, $data = array(), $create_new = true ) {
		return \EDD\SoftwareLicensing\Utils\LogRequests::log( $license_id, $site, $data, $create_new );
	}

	/**
	 * Used by get_request_checksum to sort the array by size.
	 *
	 * @since 3.5
	 * @param string $a The first item to compare for length.
	 * @param string $b The second item to compare for length.
	 *
	 * @return int The difference in length.
	 */
	private function sort_args_by_length( $a, $b ) {
		return strlen( $b ) - strlen( $a );
	}
}
