<?php
/**
 * GetVersion remote request.
 *
 * @package   EDD\SoftwareLicensing\RemoteRequests\Requests
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\RemoteRequests\Requests;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * GetVersion class.
 */
class GetVersion extends Request {

	/**
	 * Process both single and multiple licence checks.
	 *
	 * @since 3.9.0
	 * @param array $data The information of the item to check.
	 * @return array|void
	 */
	public function handle( $data ) {
		$should_return = apply_filters( 'edd_remote_latest_version_return', false );

		if ( ! isset( $data['products'] ) ) {
			$response = $this->get_single_latest_version_remote( $data );

			if ( $should_return ) {
				return $response;
			}

			$this->send_response( $response );
			return;
		}

		if ( empty( $data['products'] ) ) {
			$response['msg'] = __( 'No item provided', 'edd_sl' );

			if ( $should_return ) {
				return $response;
			}

			$this->send_response( $response );
			return;
		}

		$result = array();
		foreach ( $data['products'] as $slug => $single_data ) {
			$result[ $slug ] = $this->get_single_latest_version_remote( $single_data );
		}

		if ( $should_return ) {
			return $result;
		}

		$this->send_response( $result );
	}

	/**
	 * Process an individual license check.
	 *
	 * @since 3.7.1
	 * @param array $data The information of the item to check.
	 *
	 * @return array The item information or an error message.
	 */
	private function get_single_latest_version_remote( $data ) {

		$url       = isset( $data['url'] ) ? sanitize_text_field( urldecode( $data['url'] ) ) : false;
		$license   = isset( $data['license'] ) ? sanitize_text_field( urldecode( $data['license'] ) ) : false;
		$slug      = isset( $data['slug'] ) ? sanitize_text_field( urldecode( $data['slug'] ) ) : false;
		$item_id   = isset( $data['item_id'] ) ? absint( $data['item_id'] ) : false;
		$item_name = isset( $data['item_name'] ) ? sanitize_text_field( rawurldecode( $data['item_name'] ) ) : false;
		$beta      = isset( $data['beta'] ) ? (bool) $data['beta'] : false;
		if ( empty( $item_name ) && empty( $item_id ) ) {
			$item_name = isset( $data['name'] ) ? sanitize_text_field( rawurldecode( $data['name'] ) ) : false;
		}

		$this->log_remote_request( $license, $url, $data );

		$response = array(
			'new_version'    => '',
			'stable_version' => '',
			'sections'       => '',
			'license_check'  => '',
			'msg'            => '',
			'homepage'       => '',
			'package'        => '',
			'icons'          => array(),
			'banners'        => array(),
		);

		$bypass_name_check = defined( 'EDD_BYPASS_NAME_CHECK' ) && EDD_BYPASS_NAME_CHECK;

		if ( empty( $item_id ) && empty( $item_name ) && ! $bypass_name_check ) {
			$response['msg'] = __( 'No item provided', 'edd_sl' );

			return $response;
		}

		if ( empty( $item_id ) ) {

			if ( empty( $license ) && empty( $item_name ) ) {
				$response['msg'] = __( 'No item provided', 'edd_sl' );

				return $response;
			}

			$check_by_name_first = apply_filters( 'edd_sl_force_check_by_name', false );

			if ( empty( $license ) || $check_by_name_first ) {

				$item_id = edd_software_licensing()->get_download_id_by_name( $item_name );

			} else {

				$item_id = edd_software_licensing()->get_download_id_by_license( $license );

			}
		}

		$download = new \EDD_SL_Download( $item_id );

		if ( ! $download ) {

			if ( empty( $license ) || $check_by_name_first ) {
				/* translators: the singular download label */
				$response['msg'] = sprintf( __( 'Item name provided does not match a valid %s', 'edd_sl' ), edd_get_label_singular() );
			} else {
				/* translators: the singular download label */
				$response['msg'] = sprintf( __( 'License key provided does not match a valid %s', 'edd_sl' ), edd_get_label_singular() );
			}

			return $response;

		}

		$message               = '';
		$is_valid_for_download = edd_software_licensing()->is_download_id_valid_for_license( $download->ID, $license );
		if ( empty( $license ) ) {
			$message = __( 'No license key has been provided.', 'edd_sl' );
		} elseif ( ! $bypass_name_check && ( ! $is_valid_for_download || ( ! empty( $item_name ) && ! edd_software_licensing()->check_item_name( $download->ID, $item_name, $license ) ) ) ) {
			$download_name = ! empty( $item_name ) ? $item_name : $download->get_name();
			/* translators: the download name */
			$response['msg'] = sprintf( __( 'License key is not valid for %s', 'edd_sl' ), $download_name );

			return $response;
		}

		$stable_version = edd_software_licensing()->get_latest_version( $item_id );
		$new_version    = $stable_version;
		$slug           = ! empty( $slug ) ? $slug : $download->post_name;
		$description    = ! empty( $download->post_excerpt ) ? $download->post_excerpt : $download->post_content;
		$description    = strip_shortcodes( $description );
		$changelog      = $download->get_changelog( true );

		$download_beta = false;
		if ( $beta && $download->has_beta() ) {
			$version_beta = edd_software_licensing()->get_beta_download_version( $item_id );
			if ( version_compare( $version_beta, $stable_version, '>' ) ) {
				$changelog     = $download->get_beta_changelog();
				$new_version   = $version_beta;
				$download_beta = true;
			}
		}

		$download_requirements = $download->get_requirements();

		if ( ! empty( $download_requirements ) ) {

			$requirements_data = array();

			foreach ( $download_requirements as $platform => $min_version ) {
				$platform         = sanitize_text_field( $platform );
				$platform_version = "{$platform}_version";
				if ( ! empty( $data[ $platform_version ] ) ) {
					$requirements_data[ $platform ] = array(
						'minimum' => sanitize_text_field( $min_version ),
						'current' => sanitize_text_field( $data[ $platform_version ] ),
						'local'   => false,
					);
				}
			}

			/**
			 * Allow filtering the update requirements data, used by the EDD_SL_Requirements class.
			 *
			 * @since 3.8
			 *
			 * @param array The update requirements matched against the user's current system.
			 * @param array The requirements assigned to the download item.
			 * @param array The $data received from the item to check.
			 */
			$requirements_data = apply_filters( 'edd_sl_download_requirements_data', $requirements_data, $download_requirements, $data );

			if ( ! empty( $requirements_data ) ) {
				$requirements = new \EDD_SL_Requirements( $requirements_data );
				if ( ! $requirements->met() ) {
					$new_version = ! empty( $data['version'] )
						? sanitize_text_field( urldecode( $data['version'] ) )
						: false;
				}
			}
		}

		$response = array(
			'new_version'    => $new_version,
			'stable_version' => $stable_version,
			'name'           => $download->post_title,
			'slug'           => $slug,
			'url'            => esc_url( add_query_arg( 'changelog', '1', get_permalink( $item_id ) ) ),
			'last_updated'   => $download->post_modified_gmt,
			'homepage'       => get_permalink( $item_id ),
			'package'        => edd_software_licensing()->get_encoded_download_package_url( $item_id, $license, $url, $download_beta ),
			'download_link'  => edd_software_licensing()->get_encoded_download_package_url( $item_id, $license, $url, $download_beta ),
			'sections'       => serialize(
				array(
					'description' => wpautop( strip_tags( $description, '<p><li><ul><ol><strong><a><em><span><br>' ) ),
					'changelog'   => wpautop( strip_tags( stripslashes( $changelog ), '<p><li><ul><ol><strong><a><em><span><br>' ) ),
				)
			),
			'banners'        => serialize(
				array(
					'high' => get_post_meta( $item_id, '_edd_readme_plugin_banner_high', true ),
					'low'  => get_post_meta( $item_id, '_edd_readme_plugin_banner_low', true ),
				)
			),
			'icons'          => array(),
		);

		if ( $message ) {
			$response['msg'] = $message;
		}

		if ( has_post_thumbnail( $download->ID ) ) {
			$thumb_id  = get_post_thumbnail_id( $download->ID );
			$thumb_128 = get_the_post_thumbnail_url( $download->ID, 'sl-small' );
			if ( ! empty( $thumb_128 ) ) {
				$response['icons']['1x'] = $thumb_128;
			}

			$thumb_256 = get_the_post_thumbnail_url( $download->ID, 'sl-large' );
			if ( ! empty( $thumb_256 ) ) {
				$response['icons']['2x'] = $thumb_256;
			}
		}

		$response['icons'] = serialize( $response['icons'] );

		/**
		 * Filters the license check response.
		 *
		 * @since 1.2
		 * @since 3.7.2 Added the `$data` parameter.
		 *
		 * @param array            $response      Response.
		 * @param \EDD_SL_Download $download      Download object.
		 * @param bool             $download_beta Whether there is a beta download available.
		 * @param array            $data          Request data sent to check the item.
		 */
		$response = apply_filters( 'edd_sl_license_response', $response, $download, $download_beta, $data );

		/**
		 * Encode any emoji in the name and sections.
		 *
		 * @since 3.6.5
		 * @see https://github.com/easydigitaldownloads/EDD-Software-Licensing/issues/1313
		 */
		$response['name'] = wp_encode_emoji( $response['name'] );

		$sections             = maybe_unserialize( $response['sections'] );
		$response['sections'] = serialize( array_map( 'wp_encode_emoji', $sections ) );

		return $response;
	}
}
