<?php
/**
 * Activate remote license request.
 *
 * @package   EDD\SoftwareLicensing\RemoteRequests\Requests
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\RemoteRequests\Requests;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Activate class.
 */
class Activate extends Request {

	/**
	 * Handle the request.
	 *
	 * @since 3.9.0
	 * @param array $data
	 * @return void
	 */
	public function handle( $data ) {

		$item_id     = ! empty( $data['item_id'] ) ? absint( $data['item_id'] ) : false;
		$item_name   = ! empty( $data['item_name'] ) ? rawurldecode( $data['item_name'] ) : get_the_title( $item_id );
		$license_key = ! empty( $data['license'] ) ? urldecode( $data['license'] ) : false;

		$args = array(
			'item_name'   => $item_name,
			'key'         => $license_key,
			'url'         => isset( $data['url'] ) ? urldecode( $data['url'] ) : '',
			'item_id'     => $item_id,
			'environment' => empty( $data['environment'] ) ? 'production' : $data['environment'],
		);

		$response = array(
			'item_id'   => $item_id,
			'item_name' => $item_name,
			'license'   => 'invalid',
			'success'   => false,
		);

		$result = $this->activate( $args );
		if ( $result['success'] ) {
			$response['success'] = true;
			$response['license'] = 'valid';
		}

		$license = edd_software_licensing()->get_license( $license_key, true );
		if ( false !== $license ) {
			if ( $response['success'] ) {
				$this->log_remote_request( $license->ID, $args['url'] );
			}
			$response['expires']        = false === $license->is_lifetime ? date( 'Y-m-d H:i:s', $license->expiration ) : 'lifetime';
			$response['payment_id']     = $license->payment_id;
			$response['customer_name']  = $license->customer->name;
			$response['customer_email'] = $license->customer->email;
			$response['price_id']       = $license->price_id;
		}

		$license_id = false !== $license ? $license->ID : 0;
		$response   = apply_filters( 'edd_remote_license_activation_response', $response, $args, $license_id );

		$this->send_response( $response );
	}

	/**
	 * Activate the license.
	 *
	 * @since 3.9.0
	 * @param array $args
	 * @return array
	 */
	private function activate( $args ) {
		$activation = new \EDD\SoftwareLicensing\Licenses\Actions\Activate();

		return $activation->activate_license( $args );
	}
}
