<?php
/**
 * Class relating to license renewal handling.
 *
 * @package   EDD\SoftwareLicensing\Licenses
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Licenses;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Renewals class.
 */
class Renewals {

	/**
	 * Add a license to the cart for renewal.
	 *
	 * @since 3.9.0
	 * @param int|string $license_id_or_key License ID or key.
	 * @param bool       $by_key           Whether the license ID is a key.
	 *
	 * @return bool|\WP_Error
	 */
	public static function add_to_cart( $license_id_or_key, $by_key = false ) {
		if ( ! edd_sl_renewals_allowed() ) {
			return new \WP_Error( 'renewals_disabled', __( 'Renewals are not allowed on this site.', 'edd_sl' ) );
		}

		$license = edd_software_licensing()->get_license( $license_id_or_key, $by_key );
		if ( false === $license ) {
			return new \WP_Error( 'missing_license', __( 'No license ID supplied or invalid key provided.', 'edd_sl' ) );
		}

		if ( 'disabled' === $license->status ) {
			return new \WP_Error( 'license_disabled', __( 'The supplied license has been disabled and cannot be renewed.', 'edd_sl' ) );
		}

		$success = false;
		$order   = edd_get_order( $license->payment_id );

		if ( ! in_array( $order->status, array( 'publish', 'complete', 'partially_refunded' ), true ) ) {
			return new \WP_Error( 'payment_not_complete', __( 'The purchase record for this license is not marked as complete.', 'edd_sl' ) );
		}

		if ( empty( $license->download_id ) ) {
			return new \WP_Error( 'no_download_id', __( 'There does not appear to be a download ID attached to this license key.', 'edd_sl' ) );
		}

		if ( 'publish' !== $license->get_download()->post_status ) {
			return new \WP_Error( 'license_disabled', __( 'The download for this license is not published.', 'edd_sl' ) );
		}

		$parent_license = ! empty( $license->parent ) ? edd_software_licensing()->get_license( $license->parent ) : false;

		// This license key is part of a bundle, setup the parent.
		if ( $license->parent && false !== $parent_license ) {
			$license = $parent_license;
		}

		$options = array(
			'is_renewal'  => true,
			'license_id'  => $license->ID,
			'license_key' => $license->license_key,
		);

		$price_id = self::get_price_id( $license, $order );
		if ( ! is_null( $price_id ) ) {
			$options['price_id'] = $price_id;
		}

		// Make sure it's not already in the cart.
		$cart_key = edd_sl_get_item_position_in_cart( $license->download_id, $options );

		if ( edd_item_in_cart( $license->download_id, $options ) && false !== $cart_key ) {
			edd_remove_from_cart( $cart_key );
		}

		edd_add_to_cart( $license->download_id, $options );

		$success                          = true;
		$options['_edd_sl_renewal_added'] = true;

		// Confirm item was added to cart successfully.
		if ( ! edd_item_in_cart( $license->download_id, $options ) ) {
			return new \WP_Error( 'not_in_cart', __( 'The download for this license is not in the cart or could not be added.', 'edd_sl' ) );
		}

		// Get updated cart key.
		$cart_key = edd_sl_get_item_position_in_cart( $license->download_id, $options );

		if ( true === $success ) {

			$keys              = edd_sl_get_renewal_keys();
			$keys[ $cart_key ] = $license->license_key;

			EDD()->session->set( 'edd_is_renewal', 1 );

			$session_keys = EDD()->session->get( 'edd_renewal_keys' );

			if ( ! $session_keys || ( is_array( $session_keys ) && ! in_array( $license->license_key, $session_keys, true ) ) ) {
				EDD()->session->set( 'edd_renewal_keys', $keys );
			}

			do_action( 'edd_sl_renewals_added_to_cart', $keys );
			return true;
		}

		return new \WP_Error( 'renewal_error', __( 'Something went wrong while attempting to apply the renewal.', 'edd_sl' ) );
	}

	/**
	 * Get the price ID to use for the renewal.
	 *
	 * @since 3.9.0
	 * @param \EDD_SL_License $license
	 * @param \EDD\Orders\Order      $order
	 *
	 * @return int|null
	 */
	private static function get_price_id( $license, $order ) {
		// If the product does not have variable prices, just return null.
		if ( ! $license->get_download()->has_variable_prices() ) {
			return null;
		}

		$prices = $license->get_download()->prices;
		// If the license has a valid price ID, return it.
		if ( ! is_null( $license->price_id ) && array_key_exists( $license->price_id, $prices ) ) {
			return absint( $license->price_id );
		}

		// Look for the price ID in the original order.
		foreach ( $order->items as $order_item ) {
			if ( absint( $order_item->product_id ) !== $license->download_id ) {
				continue;
			}

			if ( absint( $order_item->cart_index ) !== absint( $license->cart_index ) ) {
				continue;
			}

			if ( ! is_null( $order_item->price_id ) && array_key_exists( $order_item->price_id, $prices ) ) {
				return absint( $order_item->price_id );
			}
		}

		$activation_limit = $license->get_activation_limit();
		// If the license's activation limit matches this price, we'll assume this price is the one that should be renewed.
		foreach ( $prices as $key => $price ) {
			if ( intval( $price['license_limit'] ) === $activation_limit ) {
				return intval( $key );
			}
		}

		// If we still haven't found a price, just return the default price.
		return edd_get_default_variable_price( $license->download_id );
	}
}
