<?php
/**
 * Check License Action
 *
 * @package   EDD\SoftwareLicensing\Licenses\Actions
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Licenses\Actions;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Check License Action
 *
 * @since 3.9.0
 */
class Check extends Action {

	/**
	 * Check the license status.
	 *
	 * @param array $args The arguments.
	 * @return string
	 */
	public function check_license( $args ) {

		$args = wp_parse_args( $args, $this->get_defaults() );
		$args = apply_filters( 'edd_sl_pre_check_license_args', $args );

		$args = array_map( 'trim', $args );

		$license = edd_software_licensing()->get_license( $args['key'], true );

		if ( false === $license ) {
			return 'invalid';
		}

		$item_name = html_entity_decode( $args['item_name'] );
		$url       = ! empty( $args['url'] ) ? $args['url'] : '';

		if ( empty( $url ) ) {
			$url = \EDD\SoftwareLicensing\Utils\URL::get_url_from_user_agent();
		}

		// Return early if the keys don't match.
		if ( $args['key'] !== $license->key ) {
			return 'invalid';
		}

		if ( ! empty( $args['item_id'] ) ) {
			if ( ! edd_software_licensing()->is_download_id_valid_for_license( $args['item_id'], $args['key'] ) ) {
				return 'invalid_item_id';
			}
		} elseif ( ( ! defined( 'EDD_BYPASS_NAME_CHECK' ) || ! EDD_BYPASS_NAME_CHECK ) && ! edd_software_licensing()->check_item_name( $license->download_id, $item_name, $license ) ) {
			return 'item_name_mismatch';
		}

		$url        = trailingslashit( edd_software_licensing()->clean_site_url( $url ) );
		$check_urls = array( $url, untrailingslashit( $url ) );

		// Now check if this is a possible IDN url and seach both the IDN and ASCII version.
		if ( function_exists( 'idn_to_utf8' ) && idn_to_utf8( $url ) !== $url ) {
			$check_urls[] = idn_to_utf8( $url );
		} elseif ( function_exists( 'idn_to_ascii' ) && idn_to_ascii( $url ) !== $url ) {
			$check_urls[] = idn_to_ascii( $url );
		}

		if ( ! $license->is_lifetime && $args['expiration'] > $license->expiration ) {
			$status = 'expired'; // this license has expired.
		} elseif ( 'disabled' === $license->status ) {
			$status = 'disabled'; // License key disabled.
		} elseif ( 'active' !== $license->status ) {
			$status = 'inactive'; // this license is not active.
		} elseif ( ! $license->is_site_active( $check_urls ) ) {
			$status = 'site_inactive';
		} else {
			do_action( 'edd_sl_check_license', $license->ID, $license->download_id );
			$status = 'valid'; // license still active.
		}

		return apply_filters( 'edd_sl_check_license_status', $status, $license );
	}
}
