<?php
/**
 * The class to process downloading a package URL from the tokenized URLs.
 *
 * @package EDD\SoftwareLicensing\Downloads
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since  3.2.4
 */

namespace EDD\SoftwareLicensing\Downloads;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Package class
 *
 * @since 3.2.4
 */
class Package implements SubscriberInterface {

	/**
	 * The file key.
	 *
	 * @var string
	 */
	private $file_id;

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_package_download' => 'process_package_download',
		);
	}

	/**
	 * Process the request for a package download
	 *
	 * @since  3.2.4
	 * @return  void
	 */
	public function process_request() {

		$data = $this->parse_url();

		if ( true === $data['success'] ) {

			// Remove the 'success' key as we do not want it in the $_GET array.
			unset( $data['success'] );

			foreach ( $data as $key => $arg ) {
				$_GET[ $key ] = $arg;
			}

			/**
			 * Fires when a package download is requested.
			 *
			 * @param array $data The data for the package download. Added in 3.9.0.
			 */
			do_action( 'edd_package_download', $data );

			// We're firing a download URL, just get out.
			wp_die();
		}

		$message = ! empty( $data['message'] ) ? $data['message'] : __( 'An error has occurred, please contact support.', 'edd_sl' );
		wp_die( esc_html( $message ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
	}

	/**
	 * Parse the URL for the package downloader
	 *
	 * @since  3.2.4
	 * @return array Array of parsed url information
	 */
	public function parse_url() {

		// Not a package download request.
		if ( false === stristr( $_SERVER['REQUEST_URI'], 'edd-sl/package_download' ) ) {
			return array();
		}

		// Assume this will be a successful parsing.
		$data = array( 'success' => true );

		$url_parts = wp_parse_url( untrailingslashit( $_SERVER['REQUEST_URI'] ) );
		$paths     = array_values( explode( '/', $url_parts['path'] ) );

		$token  = end( $paths );
		$values = explode( ':', base64_decode( $token ) );

		if ( count( $values ) !== 6 ) {
			return array(
				'success' => false,
				'message' => __( 'Invalid token supplied', 'edd_sl' ),
			);
		}

		$expires       = $values[0];
		$license_key   = $values[1];
		$download_id   = (int) $values[2];
		$url           = str_replace( '@', ':', $values[4] );
		$download_beta = (bool) $values[5];

		if ( ! edd_software_licensing()->is_download_id_valid_for_license( $download_id, $license_key ) ) {
			return array(
				'success' => false,
				'message' => __( 'Invalid license supplied', 'edd_sl' ),
			);
		}

		$license_check_args = array(
			'url'     => $url,
			'key'     => $license_key,
			'item_id' => $download_id,
		);

		$license_status = edd_software_licensing()->check_license( $license_check_args );
		switch ( $license_status ) {
			case 'valid':
				break;

			case 'expired':
				$renewal_link = add_query_arg( 'edd_license_key', $license_key, edd_get_checkout_uri() );
				$data         = array(
					'success' => false,
					'message' => sprintf(
						/* translators: %s: Renewal link */
						__( 'Your license has expired, please <a href="%s" title="Renew your license">renew it</a> to install this update.', 'edd_sl' ),
						$renewal_link
					),
				);
				break;

			case 'inactive':
			case 'site_inactive':
				$data = array(
					'success' => false,
					'message' => __( 'Your license has not been activated for this domain, please activate it first.', 'edd_sl' ),
				);
				break;

			case 'disabled':
				$data = array(
					'success' => false,
					'message' => __( 'Your license has been disabled.', 'edd_sl' ),
				);
				break;

			default:
				$data = array(
					'success' => false,
					'message' => __( 'Your license could not be validated.', 'edd_sl' ),
				);
				break;
		}

		if ( false === $data['success'] ) {
			return $data;
		}

		$computed_hash = $this->get_hash( $download_id, $license_key, $expires, $download_beta );

		if ( ! hash_equals( $computed_hash, $values[3] ) ) {
			return array(
				'success' => false,
				'message' => __( 'Provided hash does not validate.', 'edd_sl' ),
			);
		}

		$data['expires'] = $expires;
		$data['license'] = $license_key;
		$data['id']      = $download_id;
		$data['key']     = $computed_hash;
		$data['beta']    = $download_beta;

		return $data;
	}

	/**
	 * Get the encoded download package URL
	 *
	 * @since  unknown
	 * @param  int    $download_id   The Download ID.
	 * @param  string $license_key   The License Key.
	 * @param  string $url           The URL to the download package.
	 * @param  bool   $download_beta Whether to download the beta version.
	 * @return string                The encoded URL
	 */
	public function get_encoded_download_package_url( $download_id = 0, $license_key = '', $url = '', $download_beta = false ) {

		$package_url = '';

		if ( ! empty( $license_key ) ) {

			$download = new LicensedProduct( $download_id );

			$download_name = $download->get_name();
			$hours         = '+' . absint( edd_get_option( 'download_link_expiration', 24 ) ) . ' hours';
			$expires       = strtotime( $hours, current_time( 'timestamp' ) );
			$file_id       = $this->get_file_id( $download_id, $download_beta );

			$hash        = md5( $download_name . $file_id . $download_id . $license_key . (int) $expires );
			$url         = str_replace( ':', '@', $url );
			$token       = base64_encode( sprintf( '%s:%s:%d:%s:%s:%d', $expires, $license_key, $download_id, $hash, $url, $download_beta ) );
			$package_url = trailingslashit( home_url() ) . 'edd-sl/package_download/' . $token;
		}

		return apply_filters( 'edd_sl_encoded_package_url', $package_url );
	}

	/**
	 * Deliver the file download
	 *
	 * @since  3.2.4
	 * @param array $data The data for the package download.
	 * @return void
	 */
	public function process_package_download( $data ) {

		$id      = ! empty( $data['id'] ) ? absint( $data['id'] ) : false;
		$hash    = ! empty( $data['key'] ) ? urldecode( $data['key'] ) : false;
		$license = ! empty( $data['license'] ) ? sanitize_text_field( $data['license'] ) : false;
		$expires = ! empty( $data['expires'] ) ? absint( $data['expires'] ) : false;

		if ( ! $id || ! $hash || ! $license || ! $expires ) {
			wp_die( esc_html__( 'You do not have permission to download this file.', 'edd_sl' ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
		}

		$expires = is_numeric( $expires ) ? $expires : urldecode( base64_decode( $expires ) );

		do_action( 'edd_sl_before_package_download', $id, $hash, $license, $expires );

		if ( current_time( 'timestamp' ) > $expires ) {
			wp_die( esc_html__( 'Your download link has expired.', 'edd_sl' ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
		}

		if ( empty( $license ) ) {
			wp_die( esc_html__( 'No license key provided.', 'edd_sl' ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
		}

		if ( ! edd_software_licensing()->is_download_id_valid_for_license( $id, $license ) ) {
			wp_die( esc_html__( 'Invalid license supplied.', 'edd_sl' ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
		}

		$download_beta  = (bool) ! empty( $data['beta'] );
		$requested_file = $this->get_download_package( $id, $license, $hash, $expires, $download_beta );
		if ( ! $requested_file ) {
			wp_die( esc_html__( 'No download package available.', 'edd_sl' ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
		}

		$file_extension = edd_get_file_extension( $requested_file );
		$ctype          = edd_get_file_ctype( $file_extension );

		if ( ! edd_is_func_disabled( 'set_time_limit' ) ) {
			set_time_limit( 0 );
		}

		@session_write_close();
		if ( function_exists( 'apache_setenv' ) ) {
			@apache_setenv( 'no-gzip', 1 );
		}
		@ini_set( 'zlib.output_compression', 'Off' );

		nocache_headers();
		header( 'Robots: none' );
		header( 'Content-Type: ' . $ctype . '' );
		header( 'Content-Description: File Transfer' );
		header( 'Content-Disposition: attachment; filename="' . apply_filters( 'edd_requested_file_name', basename( $requested_file ) ) . '";' );
		header( 'Content-Transfer-Encoding: binary' );

		$method = edd_get_file_download_method();
		if ( 'x_sendfile' === $method && ( ! function_exists( 'apache_get_modules' ) || ! in_array( 'mod_xsendfile', apache_get_modules(), true ) ) ) {
			// If X-Sendfile is selected but is not supported, fallback to Direct.
			$method = 'direct';
		}

		if ( $this->is_absolute_path( $requested_file ) ) {

			/**
			 * Download method is set to to Redirect in settings but an absolute path was provided
			 * We need to switch to a direct download in order for the file to download properly
			 */
			$method = 'direct';
		}

		$this->log_download( $license, $id, $download_beta );

		// Redirect straight to the file.
		if ( 'redirect' === $method ) {
			header( 'Location: ' . $requested_file );
			edd_die();
		}

		$direct    = false;
		$file_path = $requested_file;

		if ( $this->is_absolute_path( $requested_file ) ) {

			/** This is an absolute path */
			$direct    = true;
			$file_path = $requested_file;

		} elseif ( defined( 'UPLOADS' ) && strpos( $requested_file, UPLOADS ) !== false ) {

			/**
			 * This is a local file given by URL so we need to figure out the path
			 * UPLOADS is always relative to ABSPATH
			 * site_url() is the URL to where WordPress is installed
			 */
			$file_path = str_replace( site_url(), '', $requested_file );
			$file_path = realpath( ABSPATH . $file_path );
			$direct    = true;

		} elseif ( false !== strpos( $requested_file, content_url() ) ) {

			/** This is a local file given by URL so we need to figure out the path */
			$file_path = str_replace( content_url(), WP_CONTENT_DIR, $requested_file );
			$file_path = realpath( $file_path );
			$direct    = true;

		} elseif ( false !== strpos( $requested_file, set_url_scheme( content_url(), 'https' ) ) ) {

			/** This is a local file given by an HTTPS URL so we need to figure out the path */
			$file_path = str_replace( set_url_scheme( content_url(), 'https' ), WP_CONTENT_DIR, $requested_file );
			$file_path = realpath( $file_path );
			$direct    = true;
		}

		// Set the file size header.
		$filesize = @filesize( $file_path );
		if ( $filesize ) {
			header( 'Content-Length: ' . $filesize );
		}

		// Now deliver the file based on the kind of software the server is running / has enabled.
		if ( stristr( getenv( 'SERVER_SOFTWARE' ), 'lighttpd' ) ) {

			header( "X-LIGHTTPD-send-file: $file_path" );

		} elseif ( $direct && ( stristr( getenv( 'SERVER_SOFTWARE' ), 'nginx' ) || stristr( getenv( 'SERVER_SOFTWARE' ), 'cherokee' ) ) ) {

			$ignore_x_accel_redirect_header = apply_filters( 'edd_ignore_x_accel_redirect', false );

			if ( ! $ignore_x_accel_redirect_header ) {
				// We need a path relative to the domain.
				$redirect_path = '/' . str_ireplace( realpath( $_SERVER['DOCUMENT_ROOT'] ), '', $file_path );
				$redirect_path = apply_filters( 'edd_sl_accel_redirect_path', $redirect_path, $file_path );
				header( "X-Accel-Redirect: $redirect_path" );
			}
		}

		if ( $direct ) {
			edd_deliver_download( $file_path );
		} else {
			// The file supplied does not have a discoverable absolute path.
			edd_deliver_download( $requested_file, true );
		}
		edd_die();
	}

	/**
	 * Deliver the package download URL
	 *
	 * @since  3.2.4
	 * @param  int    $download_id The Download ID to get the package for.
	 * @param  string $license_key The license key.
	 * @param  string $hash        The hash to verify access.
	 * @param  int    $expires     The TTL for this link.
	 * @param  bool   $download_beta Whether to download the beta version.
	 * @return string The URL for the download package.
	 */
	public function get_download_package( $download_id, $license_key, $hash, $expires = 0, $download_beta = false ) {

		$computed_hash = $this->get_hash( $download_id, $license_key, $expires, $download_beta );

		if ( ! empty( $hash ) && ! hash_equals( $computed_hash, $hash ) ) {
			wp_die( esc_html__( 'You do not have permission to download this file. An invalid hash was provided.', 'edd_sl' ), esc_html__( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
		}

		$file_url = $this->get_file_url( $download_id, $download_beta );

		return apply_filters( 'edd_sl_download_package_url', $file_url, $download_id, $license_key );
	}

	/**
	 * Check if the requested file is an absolute path.
	 *
	 * @since  3.9.0
	 * @param  string $requested_file The requested file.
	 * @return bool                   Whether the file is an absolute path.
	 */
	private function is_absolute_path( $requested_file ): bool {
		$file_details = wp_parse_url( $requested_file );
		$schemes      = array( 'http', 'https' );

		return ( ! isset( $file_details['scheme'] ) || ! in_array( $file_details['scheme'], $schemes, true ) ) && isset( $file_details['path'] ) && file_exists( $requested_file );
	}

	/**
	 * Log the download.
	 * Note that the download ID may be different from the license product ID.
	 *
	 * @since  3.9.0
	 * @param  string $license_key The license key.
	 * @param  int    $download_id The download ID.
	 * @param  bool   $download_beta Whether to download the beta version.
	 * @return int|bool            The log ID or false on failure.
	 */
	private function log_download( $license_key, $download_id, $download_beta = false ) {
		// Bail if logging is disabled.
		if ( ! apply_filters( 'edd_sl_log_package_downloads', true ) ) {
			return false;
		}

		$license = edd_software_licensing()->get_license( $license_key, true );
		if ( ! $license ) {
			return false;
		}

		$file_id = $this->get_file_id( $download_id, $download_beta );

		$args = array(
			'product_id'  => absint( $download_id ),
			'file_id'     => $file_id,
			'order_id'    => absint( $license->payment_id ),
			'price_id'    => $license->price_id,
			'customer_id' => $license->customer_id,
			'ip'          => edd_get_ip(),
		);

		if ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
			$args['user_agent'] = sanitize_text_field( $_SERVER['HTTP_USER_AGENT'] );
		}

		$log_id = edd_add_file_download_log( $args );
		if ( ! $log_id ) {
			return false;
		}

		edd_add_file_download_log_meta( $log_id, 'sl_package_download_license', $license->id );

		$file_name = $this->get_file_name( $download_id, $file_id );
		if ( ! empty( $file_name ) ) {
			edd_add_file_download_log_meta( $log_id, 'file_name', $file_name );
		}

		return $log_id;
	}

	/**
	 * Get the hash for the package download.
	 *
	 * @since  3.9.0
	 * @param  int    $download_id   The Download ID.
	 * @param  string $license_key   The License Key.
	 * @param  int    $expires       The expiration of the package URL.
	 * @param  bool   $download_beta Whether to download the beta version.
	 * @return string The hash for the package download
	 */
	private function get_hash( $download_id, $license_key, $expires, $download_beta ): string {
		$download = new LicensedProduct( $download_id );
		$file_key = $this->get_file_id( $download_id, $download_beta );

		$download_name = $download->get_name();

		/**
		 * Filter the computed hash for a package download.
		 *
		 * Allows runtime alteration of the computed hash, to allow developers to modify the validation process of a package.
		 *
		 * @since 3.6
		 *
		 * @param string  $computed_hash The current MD5 hash that was calculated.
		 * @param string  $download_name The name of the download being delivered.
		 * @param int     $file_key      The file key from the associated files from the download.
		 * @param int     $download_id   The ID of the download being delivered.
		 * @param string  $license_key   The license key being validated for the package download.
		 * @param int     $expires       The expiration of the package URL.
		 */
		return apply_filters(
			'edd_sl_package_download_computed_hash',
			md5( $download_name . $file_key . $download_id . $license_key . (int) $expires ),
			$download_name,
			$file_key,
			$download_id,
			$license_key,
			(int) $expires
		);
	}

	/**
	 * Get the file URL for the download.
	 *
	 * @since  3.9.0
	 * @param  int  $download_id   The Download ID.
	 * @param  bool $download_beta Whether to download the beta version.
	 * @return string The file URL.
	 */
	private function get_file_url( $download_id, $download_beta ): string {
		$download = new LicensedProduct( $download_id );
		$file_key = $this->get_file_id( $download_id, $download_beta );
		if ( $download_beta && $download->has_beta() ) {
			$all_files = get_post_meta( $download_id, '_edd_sl_beta_files', true );
		} else {
			$all_files = get_post_meta( $download_id, 'edd_download_files', true );
		}

		$file_url = '';
		if ( $all_files ) {
			if ( ! empty( $all_files[ $file_key ] ) && ! empty( $all_files[ $file_key ]['file'] ) ) {
				$file_url = $all_files[ $file_key ]['file'];
			} else {
				$fallback_file = array_shift( $all_files );
				if ( ! empty( $fallback_file['file'] ) ) {
					$file_url = $fallback_file['file'];
					// If the fallback file has an index, use it as the file key.
					if ( isset( $fallback_file['index'] ) ) {
						$file_key      = $fallback_file['index'];
						$this->file_id = $file_key;
					}
				}
			}
		}

		// This filter ensures compatibility with the Amazon S3 extension.
		return apply_filters( 'edd_requested_file', $file_url, $all_files, $file_key );
	}

	/**
	 * Get the file ID for the download.
	 *
	 * @since  3.9.0
	 * @param  int  $download_id   The Download ID.
	 * @param  bool $download_beta Whether to download the beta version.
	 * @return int|string The file ID, or an empty string if the meta doesn't exist.
	 */
	private function get_file_id( $download_id, $download_beta ) {
		if ( ! is_null( $this->file_id ) ) {
			return $this->file_id;
		}

		$download = new LicensedProduct( $download_id );
		if ( $download_beta && $download->has_beta() ) {
			$this->file_id = $download->get_beta_upgrade_file_key();
		} else {
			$this->file_id = $download->get_upgrade_file_key();
		}

		return $this->file_id;
	}

	/**
	 * Get the file name for the download.
	 *
	 * @since  3.9.0
	 * @param int $download_id The download ID.
	 * @param int $file_id     The file ID.
	 * @return string The file name.
	 */
	private function get_file_name( $download_id, $file_id ) {
		$file_id = absint( $file_id );
		$files   = edd_get_download_files( $download_id );

		if ( is_array( $files ) ) {
			foreach ( $files as $key => $file ) {
				if ( absint( $key ) === $file_id ) {
					return edd_get_file_name( $file );
					break;
				}
			}
		}

		return '';
	}
}
