<?php
/**
 * License Logs Table.
 *
 * @package     EDD
 * @subpackage  Database\Tables
 * @copyright   Copyright (c) 2024, Easy Digital Downloads, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Database\Tables;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Database\Table;

/**
 * Setup the global "edd_logs" database table
 *
 * @since 3.9.0
 */
final class Licenses extends Table {

	/**
	 * Table name.
	 *
	 * @access protected
	 * @since 3.9.0
	 * @var string
	 */
	protected $name = 'licenses';

	/**
	 * Database version.
	 *
	 * @access protected
	 * @since 3.9.0
	 * @var int
	 */
	protected $version = 202409241;

	/**
	 * Array of upgrade versions and methods
	 *
	 * @since 3.9.0
	 * @var array
	 */
	protected $upgrades = array(
		'202409241' => 202409241,
	);

	/**
	 * Setup the database schema.
	 *
	 * @access protected
	 * @since 3.9.0
	 * @return void
	 */
	protected function set_schema() {
		$this->schema = "
			id bigint(20) unsigned NOT NULL auto_increment,
			license_key varchar(255) default NULL,
			status varchar(20) NOT NULL default 'inactive',
			download_id bigint(20) unsigned NOT NULL default '0',
			price_id bigint(20) default NULL,
			payment_id bigint(20) default NULL,
			cart_index bigint(20) NOT NULL default '0',
			date_created datetime NOT NULL default CURRENT_TIMESTAMP,
			expiration bigint(32) default NULL,
			parent bigint(20) NOT NULL default '0',
			customer_id bigint(20) default NULL,
			user_id bigint(20) default NULL,
			PRIMARY KEY (id),
			KEY status_customer_id (status,customer_id),
			KEY id_parent (id,parent),
			KEY license_key (license_key),
			KEY payment_id_and_parent (payment_id,parent)
		";
	}

	/**
	 * Upgrade routine for version 202409241.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	public function __202409241() {

		$result = true;
		// Delete old/irrelevant database options.
		delete_option( $this->table_prefix . 'edd_licenses_db_version' );
		delete_option( 'wp_edd_licenses_db_version' );

		$checks                = array();
		$checks['price_id']    = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY COLUMN `price_id` bigint(20) default NULL;" );
		$checks['parent']      = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY COLUMN `parent` bigint(20) default '0';" );
		$checks['customer_id'] = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY COLUMN `customer_id` bigint(20) default NULL;" );
		$checks['user_id']     = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY COLUMN `user_id` bigint(20) default NULL;" );

		foreach ( $checks as $check ) {
			if ( false === $check ) {
				$result = false;
				break;
			}
		}

		return $this->is_success( $result );
	}
}
