<?php
/**
 * License Meta Table.
 *
 * @package     EDD
 * @subpackage  Database\Tables
 * @copyright   Copyright (c) 2024, Easy Digital Downloads, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Database\Tables;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Database\Table;

/**
 * Setup the global "edd_licensemeta" database table
 *
 * @since 3.9.0
 */
final class LicenseMeta extends Table {

	/**
	 * Table name
	 *
	 * @access protected
	 * @since 3.9.0
	 * @var string
	 */
	protected $name = 'licensemeta';

	/**
	 * Database version
	 *
	 * @access protected
	 * @since 3.9.0
	 * @var int
	 */
	protected $version = 202409241;

	/**
	 * Array of upgrade versions and methods
	 *
	 * @since 3.9.0
	 *
	 * @var array
	 */
	protected $upgrades = array(
		'202409241' => 202409241,
	);

	/**
	 * Setup the database schema
	 *
	 * @access protected
	 * @since 3.9.0
	 * @return void
	 */
	protected function set_schema() {
		$max_index_length = 191;
		$this->schema     = "
			meta_id bigint(20) unsigned NOT NULL auto_increment,
			edd_license_id bigint(20) unsigned NOT NULL default '0',
			meta_key varchar(255) DEFAULT NULL,
			meta_value longtext DEFAULT NULL,
			PRIMARY KEY (meta_id),
			KEY edd_license_id (edd_license_id),
			KEY meta_key (meta_key({$max_index_length}))
		";
	}

	/**
	 * Upgrade routine for version 202409241.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	public function __202409241() {

		$result = true;
		// Delete old/irrelevant database options.
		delete_option( $this->table_prefix . 'edd_licensemeta_db_version' );
		delete_option( 'wp_edd_licensemeta_db_version' );

		if ( $this->exists() && $this->column_exists( 'license_id' ) && ! $this->column_exists( 'edd_license_id' ) ) {
			$checks                         = array();
			$checks['license_id']           = $this->get_db()->query( "ALTER TABLE {$this->table_name} CHANGE `license_id` `edd_license_id` bigint(20) unsigned NOT NULL default '0';" );
			$checks['license_id_index']     = $this->get_db()->query( "ALTER TABLE {$this->table_name} DROP INDEX license_id" );
			$checks['edd_license_id_index'] = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD INDEX edd_license_id (edd_license_id)" );

			foreach ( $checks as $check ) {
				if ( false === $check ) {
					$result = false;
					break;
				}
			}
		}

		return $this->is_success( $result );
	}
}
