<?php
/**
 * Activation Query Class.
 *
 * @package   EDD\SoftwareLicensing\Database\Queries
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Database\Queries;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Database\Query;

/**
 * Class used for querying items.
 *
 * @since 3.9.0
 *
 * @see \EDD\Database\Queries\Query::__construct() for accepted arguments.
 */
class LicenseActivation extends Query {

	/** Table Properties ******************************************************/

	/**
	 * Name of the database table to query.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $table_name = 'license_activations';

	/**
	 * String used to alias the database table in MySQL statement.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $table_alias = 'la';

	/**
	 * Name of class used to setup the database schema
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $table_schema = '\\EDD\\SoftwareLicensing\\Database\\Schemas\\LicenseActivations';

	/** Item ******************************************************************/

	/**
	 * Name for a single item
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $item_name = 'license_activation';

	/**
	 * Plural version for a group of items.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $item_name_plural = 'license_activations';

	/**
	 * Callback function for turning IDs into objects
	 *
	 * @since 3.9.0
	 * @access public
	 * @var mixed
	 */
	protected $item_shape = '\\EDD\\SoftwareLicensing\\Licenses\\Activation';

	/** Cache *****************************************************************/

	/**
	 * Group to cache queries and queried items in.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $cache_group = 'license-activation';

	/**
	 * Add an activation to the database.
	 *
	 * @since 3.9.0
	 * @param array $data The data to add.
	 * @return int|bool
	 */
	public function add_item( $data = array() ) {
		if ( empty( $data['site_name'] ) ) {
			return false;
		}

		$data['site_name'] = untrailingslashit( edd_software_licensing()->clean_site_url( $data['site_name'] ) );

		return parent::add_item( $data );
	}

	/**
	 * Update an activation in the database.
	 *
	 * @since 3.9.0
	 * @param int   $item_id The activation ID.
	 * @param array $data The data to update.
	 * @return bool
	 */
	public function update_item( $item_id = 0, $data = array() ) {
		if ( empty( $item_id ) ) {
			return false;
		}
		if ( ! empty( $data['site_name'] ) ) {
			$data['site_name'] = untrailingslashit( edd_software_licensing()->clean_site_url( $data['site_name'] ) );
		}

		return parent::update_item( $item_id, $data );
	}

	/**
	 * Deletes an activation from the database.
	 * This also deletes the activation data from the license activations data table.
	 *
	 * @since 3.9.0
	 * @param int $item_id The activation ID.
	 * @return bool
	 */
	public function delete_item( $item_id = 0 ) {
		if ( empty( $item_id ) ) {
			return false;
		}

		$deleted = parent::delete_item( $item_id );
		if ( ! $deleted ) {
			return false;
		}

		$activations_data = new LicenseActivationsData();
		$data             = $activations_data->query(
			array(
				'activation_id' => $item_id,
			)
		);
		if ( ! empty( $data ) ) {
			foreach ( $data as $activation ) {
				$activations_data->delete_item( $activation->id );
			}
		}

		return $deleted;
	}

	/**
	 * Query the database for activations.
	 *
	 * @since 3.9.0
	 * @param array $args The arguments to pass to the query.
	 * @return array
	 */
	public function query( $args = array() ) {
		if ( ! empty( $args['site_name'] ) ) {
			if ( is_string( $args['site_name'] ) ) {
				$args['site_name__in'] = array( trailingslashit( $args['site_name'] ), untrailingslashit( $args['site_name'] ) );
			} else {
				$args['site_name__in'] = $args['site_name'];
			}
			unset( $args['site_name'] );
		}

		if ( ! empty( $args['site_id'] ) ) {
			if ( is_array( $args['site_id'] ) ) {
				$args['id__in'] = $args['site_id'];
			} else {
				$args['id'] = $args['site_id'];
			}
			unset( $args['site_id'] );
		}

		// Specific Licenses(s).
		if ( isset( $args['license_id'] ) && is_array( $args['license_id'] ) ) {
			$args['license_id__in'] = $args['license_id'];
			unset( $args['license_id'] );
		}

		// Specific activation status.
		if ( isset( $args['activated'] ) ) {
			if ( is_array( $args['activated'] ) ) {
				$args['activated__in'] = $args['activated'];
			}
			if ( isset( $args['activated__in'] ) ) {
				unset( $args['activated'] );
			}
		}

		// Specific is_local Checks.
		if ( isset( $args['is_local'] ) ) {
			if ( is_array( $args['is_local'] ) ) {
				$args['is_local__in'] = $args['is_local'];
				unset( $args['is_local'] );
			}
		}

		return parent::query( $args );
	}
}
