<?php
/**
 * Pass related cron events.
 *
 * @package     EDD\SoftwareLicensing\Cron\Components
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Cron\Components;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Cron\Events\SingleEvent;
use EDD\Cron\Components\Component;

/**
 * Reminders Class
 *
 * @since 3.9.0
 */
class Reminders extends Component {

	/**
	 * The unique identifier for this component.
	 *
	 * @var string
	 */
	protected static $id = 'sl_reminders';

	/**
	 * The emails object.
	 *
	 * @var \EDD_SL_Emails
	 */
	private $emails;

	/**
	 * The number of licenses to process per step.
	 *
	 * @var int
	 */
	private $per_step = 100;

	/**
	 * Gets the array of subscribed events.
	 */
	public static function get_subscribed_events(): array {
		return array(
			'edd_sl_daily_scheduled_events' => 'check_reminders',
			'edd_sl_scheduled_reminders'    => array( 'process_notice', 10, 2 ),
		);
	}

	/**
	 * Send reminders to customers.
	 *
	 * @since 3.9.0
	 */
	public function check_reminders() {
		if ( ! edd_doing_cron() ) {
			return;
		}

		foreach ( edd_software_licensing()->notices->get_notices() as $notice ) {
			if ( ! edd_software_licensing()->notices->is_enabled( $notice ) ) {
				continue;
			}

			$this->process_notice( edd_software_licensing()->notices->get_notice_id( $notice ) );
		}
	}

	/**
	 * Process an individual notice.
	 *
	 * @since 3.9.0
	 * @param string $notice_id The notice to process.
	 * @param int    $step The step to get the next set of licenses.
	 */
	public function process_notice( $notice_id, $step = 0 ) {
		$keys = $this->get_keys_for_notice( $notice_id, $step );
		if ( ! $keys ) {
			return;
		}

		foreach ( $keys as $license_id ) {

			if ( ! apply_filters( 'edd_sl_send_scheduled_reminder_for_license', true, $license_id, $notice_id ) ) {
				continue;
			}

			$license = edd_software_licensing()->get_license( $license_id );
			if ( false === $license ) {
				continue;
			}

			// Sanity check to ensure we don't send renewal notices to people with lifetime licenses.
			if ( $license->is_lifetime ) {
				continue;
			}

			$notice      = edd_software_licensing()->notices->get_notice( $notice_id );
			$send_period = edd_software_licensing()->notices->get_notice_period( $notice );
			$sent_time   = $license->get_meta( sanitize_key( '_edd_sl_renewal_sent_' . $send_period ) );
			if ( $sent_time ) {

				$expire_date = strtotime( $send_period, $sent_time );

				// The renewal period isn't expired yet so don't send again.
				if ( current_time( 'timestamp' ) < $expire_date ) {
					continue;
				}

				$license->delete_meta( sanitize_key( '_edd_sl_renewal_sent_' . $send_period ) );
			}

			$emails = $this->get_emails();
			$emails->send_renewal_reminder( $license->ID, $notice_id );
			edd_debug_log( 'Sent renewal reminder for license ID ' . $license->ID . ' for notice ID ' . $notice_id );
		}

		++$step;
		SingleEvent::add( time() + ( 5 * MINUTE_IN_SECONDS ), 'edd_sl_scheduled_reminders', array( $notice_id, $step ) );
	}

	/**
	 * Get the keys for the notice.
	 *
	 * @since 3.9.0
	 * @param string $notice_id The notice to check.
	 * @param int    $step      The step to get the next set of licenses.
	 * @return array|bool
	 */
	private function get_keys_for_notice( $notice_id, int $step ) {
		$notice = edd_software_licensing()->notices->get_notice( $notice_id );
		if ( ! edd_software_licensing()->notices->is_enabled( $notice ) ) {
			return false;
		}

		$send_period = edd_software_licensing()->notices->get_notice_period( $notice );
		// Expired notices are triggered from the set_license_status() method of EDD_Software_Licensing.
		if ( 'expired' === $send_period ) {
			return false;
		}

		return $this->get_expiring_licenses( $send_period, $step );
	}

	/**
	 * Get expiring licenses.
	 *
	 * @since 3.9.0
	 * @param string $send_period The period to check for expiring licenses.
	 * @param int    $step The step to get the next set of licenses.
	 * @return array|bool
	 */
	private function get_expiring_licenses( string $send_period, int $step ) {
		$keys = edd_software_licensing()->licenses_db->get_licenses( $this->get_args( $send_period, $step ) );
		if ( ! $keys ) {
			return false;
		}

		return $keys;
	}

	/**
	 * Get the arguments for the query.
	 *
	 * @since 3.9.0
	 * @param string $send_period The period to check for expiring licenses.
	 * @param int    $step The step to get the next set of licenses.
	 * @return array
	 */
	private function get_args( string $send_period, int $step ): array {
		$args = array(
			'number'     => $this->per_step,
			'fields'     => 'ids',
			'parent'     => 0,
			'expiration' => array(
				'start' => strtotime( $send_period . ' midnight', current_time( 'timestamp' ) ),
				'end'   => strtotime( $send_period . ' midnight', current_time( 'timestamp' ) ) + ( DAY_IN_SECONDS - 1 ),
			),
			'offset'     => $step * $this->per_step,
		);

		return apply_filters( 'edd_sl_expiring_licenses_args', $args );
	}

	/**
	 * Get the emails object.
	 *
	 * @since 3.9.0
	 * @return \EDD_SL_Emails
	 */
	private function get_emails() {
		if ( ! $this->emails ) {
			$this->emails = new \EDD_SL_Emails();
		}

		return $this->emails;
	}
}
