<?php
/**
 * Loader
 *
 * @package EDD\SoftwareLicensing\Admin\Reports
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Reports;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Class Loader
 *
 * @since 3.9.0
 * @package EDD\SoftwareLicensing\Admin\Reports
 */
class Loader implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events(): array {
		$events = array(
			'edd_pointers' => 'register_pointers',
		);
		if ( class_exists( '\\EDD\\Reports\\Endpoints\\Endpoint' ) ) {
			$events['edd_reports_init'] = array( 'register_reports', 50 );
		}

		return $events;
	}

	/**
	 * Register the pointers.
	 *
	 * @since 3.9.0
	 * @param array $pointers The pointers.
	 * @return array
	 */
	public function register_pointers( $pointers ): array {
		if ( ! edd_is_admin_page( 'reports' ) ) {
			return $pointers;
		}

		$view = filter_input( INPUT_GET, 'view', FILTER_SANITIZE_SPECIAL_CHARS );
		if ( 'file_downloads' === $view ) {
			$title   = __( 'Improved File Downloads Reporting', 'edd_sl' );
			$content = __( 'Software Licensing now logs package updates as file downloads. This means you can see a more accurate count of file downloads, including when users update your software.', 'edd_sl' );

			// If EDD is version lower than 3.1.0, add additional content.
			if ( version_compare( EDD_VERSION, '3.5.1', '<' ) ) {
				$content .= "<br /><br />" . sprintf(
					/* translators: %1$s: EDD update link, %2$s: EDD update link closing tag */
					__( 'Gain further insights into your package updates and downloads by updating to Easy Digital Downloads 3.5.1 or later. %1$sUpdate Now%2$s', 'edd_sl' ),
					'<a href="' . admin_url( 'update-core.php' ) . '">',
					'</a>'
				);
			}

			$pointers[] = array(
				'pointer_id' => 'edd_sl_file_downloads_report',
				'target'     => '#edd_general_file_downloads > h2',
				'options'    => array(
					'content'      => sprintf(
						'<h3>%s</h3><p>%s</p>',
						$title,
						$content
					),
					'position'     => array(
						'edge'  => 'top',
						'align' => 'left',
					),
					'pointerClass' => 'edd-pointer',
				),
			);
		}

		return $pointers;
	}

	/**
	 * Load the activation data report.
	 * Requires EDD endpoints to be loaded.
	 *
	 * @since 3.9.0
	 * @param mixed $reports The reports.
	 * @return void
	 */
	public function register_reports( $reports ) {
		new Renewals( $reports );
		new Upgrades( $reports );
		new ActivationData( $reports );
		new FileDownloads( $reports );
	}
}
