<?php
/**
 * Activation Data
 *
 * @package   EDD\SoftwareLicensing\Admin\Reports
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Reports;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Reports;

/**
 * Activation Data
 *
 * @since 3.9.0
 */
class ActivationData {
	use Traits\Products;

	/**
	 * The reports registry.
	 *
	 * @var \EDD\Reports\Data\Report_Registry
	 */
	private $reports;

	/**
	 * The report ID.
	 *
	 * @var string
	 */
	private $id = 'sl_activation_data';

	/**
	 * Registers the activation data reports.
	 *
	 * @param \EDD\Reports\Data\Report_Registry $reports The reports registry.
	 * @return void
	 */
	public function __construct( $reports ) {
		$this->reports = $reports;

		try {
			$this->register_report();
		} catch ( \Exception $e ) {
			// Do nothing.
		}

		add_action( 'edd_before_admin_filter_bar_reports', array( $this, 'maybe_filter_products' ) );
		add_action( 'edd_after_admin_filter_bar_reports', array( $this, 'remove_filter_products' ) );
	}

	/**
	 * Registers the activation data top level report.
	 *
	 * @since 3.9.0
	 */
	public function register_report() {
		$tiles  = $this->get_tiles();
		$charts = $this->get_charts();

		$this->reports->add_report(
			$this->id,
			array(
				'label'     => __( 'Activation Data', 'edd_sl' ),
				'icon'      => 'admin-site-alt3',
				'priority'  => 60,
				'endpoints' => array(
					'tiles'  => array_keys( $tiles ),
					'charts' => array_keys( $charts ),
				),
				'filters'   => array(
					'dates',
					'products',
				),
			)
		);

		foreach ( $tiles as $tile => $class ) {
			new $class( $this->reports );
		}
		foreach ( $charts as $chart => $class ) {
			if ( is_array( $class ) ) {
				new $class[0]( $this->reports, $class[1] );
			} else {
				new $class( $this->reports );
			}
		}
	}

	/**
	 * Returns the tiles for the activation data report.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function get_tiles() {
		return array(
			'sl_new_activations'         => 'EDD\SoftwareLicensing\Admin\Reports\Endpoints\Tiles\NewActivations',
			'sl_most_recent_api_request' => 'EDD\SoftwareLicensing\Admin\Reports\Endpoints\Tiles\RecentRequest',
		);
	}

	/**
	 * Gets the charts for the activation data report.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function get_charts() {
		$charts = array();
		if ( ! empty( $this->get_product() ) ) {
			$charts['sl_product_versions'] = 'EDD\SoftwareLicensing\Admin\Reports\Endpoints\Pies\Versions';
		}
		$meta_keys = eddsl_get_activation_data_meta_keys();
		if ( ! empty( $meta_keys ) ) {
			foreach ( $meta_keys as $key => $label ) {
				$charts[ 'sl_meta_key_' . $key ] = array( 'EDD\SoftwareLicensing\Admin\Reports\Endpoints\Pies\Meta', $key );
			}
		}
		$charts['sl_activation_line_graph'] = 'EDD\SoftwareLicensing\Admin\Reports\Endpoints\Charts\Activations';
		$charts['sl_all_requests']          = 'EDD\SoftwareLicensing\Admin\Reports\Endpoints\Charts\AllRequests';

		return $charts;
	}

	/**
	 * Gets the product for the reports.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function get_product() {
		$product = Reports\get_filter_value( 'products' );

		return 'all' === $product ? '' : $product;
	}
}
