<?php
/**
 * Handle refunds for software licensing.
 *
 * @package   EDD\SoftwareLicensing\Admin\Orders
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Orders;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Refunds class.
 */
class Refunds implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_after_submit_refund_table' => array( 'show_refund_checkbox', 50 ),
			'edd_refund_order'              => 'maybe_revoke_licenses',
		);
	}

	/**
	 * Show the refund checkbox for licenses.
	 *
	 * @param \EDD\Orders\Order $order Order object.
	 */
	public function show_refund_checkbox( \EDD\Orders\Order $order ) {
		$licenses = edd_software_licensing()->get_licenses_of_purchase( $order->id );
		if ( empty( $licenses ) ) {
			return;
		}

		?>
		<fieldset class="edd-form-group edd-sl-license-refund-options">
			<legend class="screen-reader-text"><?php esc_html_e( 'License Refund Options', 'edd_sl' ); ?></legend>
			<?php
			foreach ( $licenses as $license ) {
				$order_items   = edd_get_order_items(
					array(
						'cart_index' => $license->cart_index,
						'order_id'   => $order->id,
						'product_id' => $license->download_id,
					)
				);
				$order_item_id = ! empty( $order_items ) ? $order_items[0]->id : 0;
				$classes       = array( 'edd-form-group__control', 'edd-hidden' );
				if ( $order_item_id ) {
					$classes[] = 'edd-sl-license-order-item-' . $order_item_id;
				}
				?>
				<div class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>">
					<input
						type="checkbox"
						id="edd_sl__revoke-license-<?php echo esc_attr( $license->id ); ?>"
						name="edd_sl__revoke-license[<?php echo esc_attr( $license->id ); ?>]"
						class="edd-form-group__input"
						value="<?php echo esc_attr( $license->id ); ?>"
						<?php disabled( 'disabled', $license->status ); ?>
					>
					<label for="edd_sl__revoke-license-<?php echo esc_attr( $license->id ); ?>" class="edd-form-group__label">
						<?php
						printf(
							/* translators: %s: License key */
							esc_html__( 'Revoke license key %1$s for %2$s', 'edd_sl' ),
							esc_html( $license->license_key ),
							esc_html( edd_get_download_name( $license->download_id, $license->price_id ) )
						);
						?>
					</label>
				</div>
				<?php
			}
			?>
		</fieldset>
		<?php
	}

	/**
	 * Maybe revoke licenses when processing a refund from the admin.
	 *
	 * @param int $order_id Order ID.
	 */
	public function maybe_revoke_licenses( $order_id ) {
		if ( ! edd_doing_ajax() ) {
			return;
		}

		if ( ! current_user_can( 'edit_shop_payments', $order_id ) ) {
			return;
		}

		if ( empty( $_POST['data'] ) ) {
			return;
		}

		// Get our data out of the serialized string.
		parse_str( $_POST['data'], $form_data );

		if ( empty( $form_data['edd_sl__revoke-license'] ) ) {
			return;
		}

		foreach ( $form_data['edd_sl__revoke-license'] as $license_id ) {
			$license = edd_software_licensing()->get_license( $license_id );
			if ( empty( $license ) ) {
				continue;
			}

			$license->revoke();
		}
	}
}
