<?php
/**
 * Versions section.
 *
 * @package EDD\SoftwareLicensing\Admin\Downloads\Editor
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads\Editor;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\Downloads\Editor\Section;
use EDD\HTML;

/**
 * Versions section.
 *
 * @since 3.9.0
 */
class Versions extends Section {

	/**
	 * Section ID.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $id = 'versions';

	/**
	 * Section priority.
	 *
	 * @since 3.9.0
	 * @var int
	 */
	protected $priority = 2;

	/**
	 * Section icon.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $icon = 'update';

	/**
	 * Supports.
	 *
	 * @since 3.9.0
	 * @var array
	 */
	protected $supports = array(
		'product-type' => array(
			'bundle'     => 'false',
			'all_access' => 'false',
			'service'    => 'false',
		),
	);

	/**
	 * Get the section label.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_label() {
		return esc_html__( 'Versions', 'edd_sl' );
	}

	/**
	 * Render the section.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	public function render() {
		$download = $this->item;
		if ( ! current_user_can( 'edit_product', $download->ID ) ) {
			return;
		}

		$version = get_post_meta( $download->ID, '_edd_sl_version', true );
		$file    = get_post_meta( $download->ID, '_edd_sl_upgrade_file_key', true );
		?>
		<div class="edd-form-group">
			<?php do_action( 'edd_sl_license_metabox_before_version', $download->ID ); ?>
			<label for="edd_sl_version" class="edd-form-group__label">
				<?php esc_html_e( 'Version Number', 'edd_sl' ); ?>
				<?php
				$tooltip = new HTML\Tooltip(
					array(
						'title'   => __( 'Version Number', 'edd_sl' ),
						'content' => __( 'Version numbers should follow Semantic Versioning (SemVer) standards. For example, 1.0.0, 1.0.1, 1.2.0, etc.', 'edd_sl' ),
					)
				);
				$tooltip->output();
				?>
			</label>
			<?php
			$version_input = new HTML\Text(
				array(
					'name'  => 'edd_sl_version',
					'id'    => 'edd_sl_version',
					'value' => $version,
					'class' => 'edd-form-group__input',
				)
			);
			$version_input->output();
			?>
			<p class="edd-form-group__help description">
				<?php esc_html_e( 'Enter the current version number.', 'edd_sl' ); ?>
			</p>
			<?php do_action( 'edd_sl_license_metabox_after_version', $download->ID ); ?>
		</div>

		<div class="edd-form-group">
			<?php do_action( 'edd_sl_license_metabox_before_upgrade_file', $download->ID ); ?>
			<label for="edd_sl_upgrade_file"><?php esc_html_e( 'Update File', 'edd_sl' ); ?></label>
			<div class="edd-form-group__control">
				<select name="edd_sl_upgrade_file" id="edd_sl_upgrade_file">
					<?php
					$files = get_post_meta( $download->ID, 'edd_download_files', true );
					if ( is_array( $files ) && ! array_key_exists( $file, $files ) ) {
						echo '<option value="-1" >' . esc_html__( 'No update file selected', 'edd_sl' ) . '</option>';
					}
					if ( is_array( $files ) ) {
						foreach ( $files as $key => $value ) {
							$name = isset( $files[ $key ]['name'] ) ? $files[ $key ]['name'] : '';
							echo '<option value="' . esc_attr( $key ) . '" ' . selected( $key, $file, false ) . '>' . esc_html( $name ) . '</option>';
						}
					}
					?>
				</select>
				<p class="edd-form-group__help description"><?php esc_html_e( 'Choose the source file to be used for automatic updates.', 'edd_sl' ); ?></p>
				<?php do_action( 'edd_sl_license_metabox_after_upgrade_file', $download->ID ); ?>
			</div>
		</div>

		<div class=" edd-form-group">
			<?php do_action( 'edd_sl_license_metabox_before_changelog', $download->ID ); ?>
			<label class="edd-form-group__label" for="edd_sl_changelog"><?php esc_html_e( 'Changelog', 'edd_sl' ); ?></label>
			<div class="edd-form-group__control">
				<?php
				$changelog = get_post_meta( $download->ID, '_edd_sl_changelog', true );
				wp_editor(
					stripslashes( $changelog ),
					'edd_sl_changelog',
					array(
						'textarea_name' => 'edd_sl_changelog',
						'media_buttons' => false,
						'teeny'         => true,
						'textarea_rows' => 15,
					)
				);
				?>
			</div>
			<p class="edd-form-group__help description"><?php esc_html_e( 'Enter details about what changed.', 'edd_sl' ); ?></p>
			<?php do_action( 'edd_sl_license_metabox_after_changelog', $download->ID ); ?>
		</div>

		<?php
		$upgrade_notice = get_post_meta( $download->ID, '_edd_sl_upgrade_notice', true );
		?>
		<div class="edd-form-group">
			<label class="edd-form-group__label" for="edd_sl_upgrade_notice"><?php esc_html_e( 'Upgrade Notice', 'edd_sl' ); ?></label>
			<div class="edd-form-group__control">
				<textarea name="edd_sl_upgrade_notice" id="edd_sl_upgrade_notice" class="edd-sl-keys-input" rows="5"><?php echo esc_textarea( stripslashes( $upgrade_notice ) ); ?></textarea>
			</div>
			<p class="edd-form-group__help description"><?php esc_html_e( 'Enter the notice that will be displayed to customers when a new version is available.', 'edd_sl' ); ?></p>
		</div>
		<?php
		$sr_enabled      = (bool) get_post_meta( $download->ID, 'edd_sr_enabled', true );
		$batch_enabled   = (bool) get_post_meta( $download->ID, 'edd_sr_batch_enabled', true );
		$batch_max       = get_post_meta( $download->ID, 'edd_sr_batch_max', true );
		$version_enabled = (bool) get_post_meta( $download->ID, 'edd_sr_version_enabled', true );
		$version         = get_post_meta( $download->ID, 'edd_sr_version', true );
		$is_above_below  = empty( get_post_meta( $download->ID, 'edd_sr_version_limit', true ) );
		?>
		<div class="edd-form-group">
			<div class="edd-form-group__label">
				<?php esc_html_e( 'Staged Rollouts', 'edd_sl' ); ?>
				<?php
				$tooltip = new HTML\Tooltip(
					array(
						'title'   => __( 'Staged Rollouts', 'edd_sl' ),
						'content' => __( 'A Staged Rollout is a method of releasing updates to your software to a limited set of customers, as opposed to releasing the update to all of your customers at the same time. <br /><br /><strong>How does this work?</strong> Staged rollouts uses the Software Licensing update API to determine which installations are shown the update, based on the conditions. Customers will be able to download the staged version from your website immediately for fresh installs.', 'edd_sl' ),
					)
				);
				$tooltip->output();
				?>
			</div>
			<div class="edd-form-group__control edd-toggle">
				<input
					type="checkbox"
					name="edd_sr_rollouts_enabled"
					id="edd_sr_rollouts_enabled"
					class="edd-form-group__input edd-requirement"
					value="1"
					<?php checked( true, $sr_enabled ); ?>
					data-edd-requirement="staged-rollouts"
				/>&nbsp;
				<label for="edd_sr_rollouts_enabled"><?php esc_html_e( 'Enable staged rollouts.', 'edd_sl' ); ?></label>
			</div>
		</div>

		<div data-edd-requires-staged-rollouts="true">
			<div class="edd-form-group__label"><?php esc_html_e( 'How do you want to limit the distribution of new version?', 'edd_sl' ); ?></div>

			<div class="edd-form-group">
				<div class="edd-form-group__control edd-toggle">
					<input
						type="checkbox"
						name="edd_sr_batch_enabled"
						id="edd_sr_batch_enabled"
						class="edd-form-group__input edd-requirement"
						value="1"
						<?php checked( true, $batch_enabled ); ?>
						data-edd-requirement="staged-rollouts-percentage"
					/>&nbsp;
					<label for="edd_sr_batch_enabled"><?php esc_html_e( 'Release to a percentage of active installations.', 'edd_sl' ); ?></label>
					<?php
					$tooltip = new HTML\Tooltip(
						array(
							'title'   => __( 'Percentage of Active Installations', 'edd_sl' ),
							'content' => implode(
								'<br /><br />',
								array(
									__( 'When enabled, the update will be released to sites until the percentage of active installations has been reached.', 'edd_sl' ),
									__( 'Once reached, the release will no longer distribute the new version until you increase this percentage.', 'edd_sl' ),
									__( 'Complete your staged release by setting this value to 100% or disabling the Staged Rollout setting.', 'edd_sl' ),
								)
							),
						)
					);
					$tooltip->output();
					?>
				</div>
			</div>

			<div class="edd-form-group" data-edd-requires-staged-rollouts-percentage="true">
				<div class="edd-form-group__control edd-range">
					<?php $batch_max = ! empty( $batch_max ) ? $batch_max : 25; ?>
					<div id="edd-sl-batch-slider" class="edd-range__slider" data-max="100" data-min="0" data-value="<?php echo absint( $batch_max ); ?>"></div>
					<div class="edd-amount-type-wrapper">
						<?php
						$number_input = new HTML\Number(
							array(
								'name'  => 'edd_sr_batch_max',
								'id'    => 'edd_sr_batch_max',
								'class' => array( 'edd-range__input', 'edd-amount-input', 'small-text', 'no-controls' ),
								'max'   => 100,
								'min'   => 0,
								'step'  => 1,
								'value' => absint( $batch_max ),
							)
						);
						$number_input->output();
						?>
						<span class="edd-input__symbol edd-input__symbol--suffix">%</span>
					</div>
				</div>
			</div>

			<div class="edd-form-group">
				<div class="edd-form-group__control edd-toggle">
					<input
						type="checkbox"
						name="edd_sr_version_enabled"
						id="edd_sr_version_enabled"
						class="edd-form-group__input edd-requirement"
						value="1"
						<?php checked( true, $version_enabled ); ?>
						data-edd-requirement="staged-rollouts-version-requirement"
					/>&nbsp;
					<label for="edd_sr_version_enabled"><?php esc_html_e( 'Limit the release to users based on their currently installed version.', 'edd_sl' ); ?></label>
					<?php
					$tooltip = new HTML\Tooltip(
						array(
							'title'   => __( 'Version Requirement', 'edd_sl' ),
							'content' => __( 'Only distribute the update to users based on their currently installed version using the less than or greater than options below.', 'edd_sl' ),
						)
					);
					$tooltip->output();
					?>
				</div>
			</div>

			<div class="edd-form-group" data-edd-requires-staged-rollouts-version-requirement="true">
				<div class="edd-form-group__control">
					<span class="edd-amount-type-wrapper">
						<?php
						$version_limit_input = new HTML\Select(
							array(
								'name'             => 'edd_sr_version_limit',
								'id'               => 'edd_sr_version_limit_below',
								'class'            => 'edd-form-group__input edd__input edd__input--left',
								'show_option_none' => false,
								'show_option_all'  => false,
								'options'          => array(
									'0' => __( 'Less than or equal to', 'edd_sl' ),
									'1' => __( 'Greater than or equal to', 'edd_sl' ),
								),
							)
						);
						$version_limit_input->output();

						$version_input = new HTML\Text(
							array(
								'name'           => 'edd_sr_version',
								'id'             => 'edd_sr_version',
								'value'          => $version,
								'class'          => array( 'edd-form-group__input', 'edd__input', 'edd__input--right' ),
								'placeholder'    => __( 'Enter a version number', 'edd_sl' ),
								'data-edd-value' => $version,
								'include_span'   => false,
							)
						);
						$version_input->output();
						?>
					</span>
				</div>
			</div>
		</div>
		<?php
	}
}
