<?php
/**
 * Preset Keys section.
 *
 * @package EDD\SoftwareLicensing\Admin\Downloads\Editor
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads\Editor;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\Downloads\Editor\Section;

/**
 * Preset Keys section.
 *
 * @since 3.9.0
 */
class PresetKeys extends Section {

	/**
	 * Section ID.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $id = 'preset-keys';

	/**
	 * Section priority.
	 *
	 * @since 3.9.0
	 * @var int
	 */
	protected $priority = 10;

	/**
	 * Section icon.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $icon = 'admin-network';

	/**
	 * Get the section label.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_label() {
		$label = esc_html__( 'Preset Keys', 'edd_sl' );

		if ( 'download' === get_post_type( $this->item->id ) ) {
			$preset_keys = get_post_meta( $this->item->id, '_edd_sl_keys', true );

			if ( empty( $preset_keys ) ) {
				return $label;
			}

			$count = count( explode( "\n", $preset_keys ) );
			if ( $count > 0 && $count < 99 ) {
				$label .= ' <span class="edd-round edd-number">' . $count . '</span>';
			} elseif ( $count >= 99 ) {
				$label .= ' <span class="edd-round edd-number">' . __( '99+', 'edd_sl' ) . '</span>';
			}
		}

		return $label;
	}

	/**
	 * Render the section.
	 *
	 * @since 3.9.0
	 */
	public function render() {
		$download = $this->item;
		if ( ! current_user_can( 'edit_product', $download->ID ) ) {
			return;
		}

		$keys = get_post_meta( $download->ID, '_edd_sl_keys', true );
		?>
		<div class="edd-form-group">
			<?php do_action( 'edd_sl_license_metabox_before_license_keys', $download->ID ); ?>
			<label class="edd-form-group__label" for="edd_sl_keys"><?php esc_html_e( 'Preset License Keys', 'edd_sl' ); ?></label>
			<div class="edd-form-group__control">
				<textarea name="edd_sl_keys" class="edd-sl-keys-input" id="edd_sl_keys" rows="10"><?php echo esc_textarea( stripslashes( $keys ) ); ?></textarea>
			</div>
			<p class="edd-form-group__help description"><?php esc_html_e( 'Enter available license keys, one per line. If empty, keys will be automatically generated. ', 'edd_sl' ); ?></p>
			<?php do_action( 'edd_sl_license_metabox_after_license_keys', $download->ID ); ?>
		</div>

		<div class="edd-form-group" data-sl-supports-bundle="false">
			<div class="edd-form-group__control edd-toggle">
				<input type="checkbox" name="edd_sl_no_keys_behavior" id="edd_sl_no_keys_behavior" value="1" <?php checked( 'draft', get_post_meta( $download->ID, '_edd_sl_no_keys_behavior', true ) ); ?>/>
				<label for="edd_sl_no_keys_behavior"><?php esc_html_e( 'Set this download to a draft when all preset keys have been distributed.', 'edd_sl' ); ?></label>
			</div>
			<p class="edd-form-group__help description"><?php esc_html_e( 'Prevent users from purchasing this download when there are no more preset keys.', 'edd_sl' ); ?></p>
		</div>

		<div class="edd-form-group">
			<p class="edd-form-group__help description">
				<span class="dashicons dashicons-lightbulb"></span>
				<strong><?php esc_html_e( 'Did you know?', 'edd_sl' ); ?></strong>
				<?php
				printf(
					/* translators: %1$s: opening anchor tag, %2$s: closing anchor tag */
					esc_html__( 'You can be %1$snotified by email%2$s when you are running low on preset keys.', 'edd_sl' ),
					'<a href="' . esc_url(
						edd_get_admin_url(
							array(
								'page'  => 'edd-emails',
								'email' => 'sl_low_keys',
							)
						)
					) . '" target="_blank">',
					'</a>'
				);
				?>
			</p>
		</div>

		<?php
	}
}
