<?php
/**
 * Betas section.
 *
 * @package EDD\SoftwareLicensing\Admin\Downloads\Editor
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads\Editor;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\Downloads\Editor\Section;
use EDD\HTML;

/**
 * Betas section.
 *
 * @since 3.9.0
 */
class Betas extends Section {

	/**
	 * Section ID.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $id = 'betas';

	/**
	 * Section priority.
	 *
	 * @since 3.9.0
	 * @var int
	 */
	protected $priority = 3;

	/**
	 * Section icon.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $icon = 'flag';

	/**
	 * Get the section label.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_label() {
		return esc_html__( 'Betas', 'edd_sl' );
	}

	/**
	 * Supports.
	 *
	 * @since 3.9.0
	 * @var array
	 */
	protected $supports = array(
		'product-type' => array(
			'bundle'     => 'false',
			'all_access' => 'false',
			'service'    => 'false',
		),
	);

	/**
	 * Render the section.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	public function render() {
		$download = $this->item;
		if ( ! current_user_can( 'edit_product', $download->ID ) ) {
			return;
		}

		$betas_enabled  = (bool) get_post_meta( $download->ID, '_edd_sl_beta_enabled', true );
		$beta_version   = get_post_meta( $download->ID, '_edd_sl_beta_version', true );
		$beta_changelog = get_post_meta( $download->ID, '_edd_sl_beta_changelog', true );
		?>
		<div class="edd-form-group">
			<div class="edd-form-group__control edd-toggle">
				<input type="hidden" name="edd_sl_beta_enabled" value="0" />
				<input
					type="checkbox"
					name="edd_sl_beta_enabled"
					id="edd_sl_beta_enabled"
					class="edd-requirement"
					value="1"
					<?php checked( true, $betas_enabled ); ?>
					data-edd-requirement="enable-beta"
				/>
				<label for="edd_sl_beta_enabled">
					<?php
					/* translators: %s: singular download label */
					printf( esc_html__( 'Enable a beta version of this %s.', 'edd_sl' ), esc_html( edd_get_label_singular( true ) ) );
					?>
				</label>
				<?php
				$tooltip = new HTML\Tooltip(
					array(
						'title'   => __( 'Beta Version', 'edd_sl' ),
						'content' => __( 'Configure your beta version to deliver beta updates to any users who have opted in.', 'edd_sl' ),
					)
				);
				$tooltip->output();
				?>
			</div>
			<p class="edd_sl_beta_docs_link">
				<?php
				printf(
					/* translators: %1$s: Link to beta documentation, %2$s: Link closing tag */
					__( 'For more information on releasing beta versions, please %1$sreview our documentation%2$s.', 'edd_sl' ),
					sprintf(
						'<a href="%s" target="_blank">',
						esc_url(
							edd_link_helper(
								'https://easydigitaldownloads.com/docs/software-licensing-releasing-beta-versions/',
								array(
									'utm_medium'  => 'edit-download',
									'utm_content' => 'releasing-beta-versions',
								)
							)
						)
					),
					'</a>'
				)
				?>
			</p>
		</div>

		<div class="edd-form-group" data-edd-requires-enable-beta="true">
			<label><?php esc_html_e( 'Beta Version File', 'edd_sl' ); ?></label>
			<div id="edd_sl_beta_files_wrap">
				<?php
				$files = get_post_meta( $download->ID, '_edd_sl_beta_files', true );
				if ( empty( $files ) ) {
					$files = array(
						1 => array(
							'name' => '',
							'file' => '',
						),
					);
				}
				foreach ( $files as $key => $value ) :
					?>
					<div class="edd-form-row edd-repeatable-row-standard-fields edd_repeatable_upload_wrapper">
						<div class="edd-form-group edd-file-name">
							<label for="edd_sl_beta_files[<?php echo esc_attr( $key ); ?>][name]" class="edd-form-group__label edd-repeatable-row-setting-label">
								<?php esc_html_e( 'File Name', 'edd_sl' ); ?>
							</label>
							<div class="edd-form-group__control">
								<?php
								$text = new HTML\Text(
									array(
										'name'  => 'edd_sl_beta_files[' . esc_attr( $key ) . '][name]',
										'id'    => 'edd_sl_beta_files[' . esc_attr( $key ) . '][name]',
										'value' => $value['name'],
										'class' => 'edd-form-group__input edd_repeatable_name_field large-text',
									)
								);
								$text->output();
								?>
							</div>
						</div>

						<div class="edd-form-group edd-file-url">
							<label for="edd_sl_beta_files[<?php echo esc_attr( $key ); ?>][file]" class="edd-form-group__label edd-repeatable-row-setting-label">
								<?php esc_html_e( 'File URL', 'edd_sl' ); ?>
							</label>
							<div class="edd-form-group__control edd_repeatable_upload_field_container">
								<?php
								$file = new HTML\Text(
									array(
										'name'        => 'edd_sl_beta_files[' . esc_attr( $key ) . '][file]',
										'id'          => 'edd_sl_beta_files[' . esc_attr( $key ) . '][file]',
										'value'       => $value['file'],
										'placeholder' => __( 'Enter, upload, choose from Media Library', 'edd_sl' ),
										'class'       => 'edd-form-group__input edd_repeatable_upload_field edd_upload_field large-text',
									)
								);
								$file->output();
								?>

								<span class="edd_upload_file">
									<button type="button" data-uploader-title="<?php esc_attr_e( 'Select Files', 'edd_sl' ); ?>" data-uploader-button-text="<?php esc_attr_e( 'Select', 'edd_sl' ); ?>" class="edd_upload_file_button" onclick="return false;">
										<span class="dashicons dashicons-admin-links"></span>
										<span class="screen-reader-text"><?php esc_html_e( 'Select Files', 'edd_sl' ); ?></span>
									</button>
								</span>
							</div>
						</div>
					</div>
				<?php endforeach; ?>
				<input type="hidden" name="edd_sl_beta_upgrade_file" value="1"/>
				<p class="edd-form-group__help description">
					<?php esc_html_e( 'Choose the source file to be used for automatic update to beta.', 'edd_sl' ); ?>
				</p>
			</div>
		</div>

		<div class="edd-form-group" data-edd-requires-enable-beta="true">
			<label for="edd_sl_beta_version">
				<?php esc_html_e( 'Beta Version', 'edd_sl' ); ?>
				<?php
				$tooltip = new HTML\Tooltip(
					array(
						'title'   => __( 'Entering a Beta Version', 'edd_sl' ),
						'content' => __( 'Version numbers should follow Semantic Versioning (SemVer) standards. For example, 1.0.0-beta.1, 1.0.0-beta.2, etc.', 'edd_sl' ),
					)
				);
				$tooltip->output();
				?>
			</label>
			<?php
			$version_input = new HTML\Text(
				array(
					'name'        => 'edd_sl_beta_version',
					'id'          => 'edd_sl_beta_version',
					'value'       => $beta_version,
					'class'       => 'edd-form-group__input',
					'placeholder' => __( 'Enter beta version', 'edd_sl' ),
				)
			);
			$version_input->output();
			?>

		</div>

		<div class="edd-form-group" data-edd-requires-enable-beta="true">
			<label for="edd_sl_beta_changelog"><?php esc_html_e( 'Beta Changelog', 'edd_sl' ); ?></label>
			<?php
			wp_editor(
				stripslashes( $beta_changelog ),
				'edd_sl_beta_changelog',
				array(
					'textarea_name' => 'edd_sl_beta_changelog',
					'media_buttons' => false,
					'teeny'         => true,
					'textarea_rows' => 15,
				)
			);
			?>
		</div>
		<?php
	}
}
