<?php
/**
 * EDD Software Licensing Emails class.
 *
 * @package   EDD\SoftwareLicensing\Emails
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/**
 * EDD_SL_Emails class.
 */
class EDD_SL_Emails {

	/**
	 * @var bool If true, then exceptions will be thrown on failures.
	 * @since 3.8.3
	 */
	protected $throw_exceptions = false;

	/**
	 * Enables exceptions on errors.
	 *
	 * @since 3.8.3
	 *
	 * @return $this
	 */
	public function with_exceptions() {
		$this->throw_exceptions = true;

		return $this;
	}

	/**
	 * Sends a renewal reminder email to the customer.
	 *
	 * @since <unknown>
	 *
	 * @param int $license_id The license ID.
	 * @param int $notice_id  The notice ID.
	 *
	 * @return bool
	 * @throws \Exception If an exception is thrown.
	 */
	public function send_renewal_reminder( $license_id = 0, $notice_id = 0 ) {

		if ( empty( $license_id ) ) {
			if ( $this->throw_exceptions ) {
				throw new \Exception( __( 'Reminder not sent: no license key provided.', 'edd_sl' ) );
			}

			return false;
		}

		$notice = edd_software_licensing()->notices->get_notice( $notice_id );
		if ( ! $notice || ! edd_software_licensing()->notices->is_enabled( $notice ) ) {
			if ( $this->throw_exceptions ) {
				throw new \Exception( esc_html__( 'Reminder not sent: renewal reminders are not enabled.', 'edd_sl' ) );
			}

			return false;
		}

		$send    = true;
		$license = edd_software_licensing()->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		$exception_message = __( 'Reminder not sent: unexpected sending failure.', 'edd_sl' );

		if ( $license->is_lifetime ) {
			$exception_message = __( 'License never expires.', 'edd_sl' );
			$send              = false;
		}

		if ( $this->is_unsubscribed( $license ) ) {
			$exception_message = __( 'Reminder not sent: customer is not subscribed to reminder emails.', 'edd_sl' );
			$send              = false;
		}

		if ( 'disabled' === $license->status ) {
			$exception_message = __( 'Reminder not sent: this license key is disabled.', 'edd_sl' );
			$send              = false;
		}

		$send = apply_filters( 'edd_sl_send_renewal_reminder', $send, $license->ID, $notice_id );

		if ( ! $license || ! $send || ! empty( $license->parent ) ) {
			if ( $this->throw_exceptions ) {
				throw new \Exception( $exception_message );
			}

			return false;
		}

		$email_object = edd_get_email( $notice_id );
		$email        = EDD\Emails\Registry::get( $notice_id, array( $license->ID, $license, $email_object ) );
		$sent         = $email->send();
		if ( $sent ) {
			$period = edd_get_email_meta( $email->email->id, 'period', true );
			$license->update_meta( sanitize_key( '_edd_sl_renewal_sent_' . $period ), time() ); // Prevent renewal notices from being sent more than once.
		}

		if ( ! $sent && $this->throw_exceptions ) {
			throw new \Exception( esc_html__( 'Reminder not sent: email failed to send.', 'edd_sl' ) );
		}

		if ( $sent ) {
			$log_id = $license->add_log( __( 'LOG - Renewal Notice Sent', 'edd_sl' ), __( 'Sent via the send_renewal_reminder method.', 'edd_sl' ), 'renewal_notice' );
			add_post_meta( $log_id, '_edd_sl_renewal_notice_id', $notice_id );
			$license->update_meta( sanitize_key( '_edd_sl_renewal_sent_' . $period ), time() ); // Prevent renewal notices from being sent more than once.
		}

		return $sent;
	}

	/**
	 * Filter the reminder template tags.
	 *
	 * @deprecated 3.9.0
	 * @param string          $text The text to filter.
	 * @param int             $license_id The license ID.
	 * @param \EDD_SL_License $license The license object.
	 * @return string
	 */
	public function filter_reminder_template_tags( $text = '', $license_id = 0, $license = null ) {
		if ( ! $license ) {
			$license = edd_software_licensing()->get_license( $license_id );
		}
		if ( false === $license ) {
			return $text;
		}

		return apply_filters( 'edd_sl_renewal_message', edd_do_email_tags( $text, $license->ID, $license, 'license' ), $license->ID );
	}

	/**
	 * Determine if email notifications for this license are disabled
	 *
	 * @since  3.5.11
	 *
	 * @param \EDD_SL_License $license EDD_SL_License object.
	 *
	 * @return bool
	 */
	public function is_unsubscribed( \EDD_SL_License $license ) {
		return (bool) $license->get_meta( 'edd_sl_unsubscribed', true );
	}

	/**
	 * Renders a preview for a renewal email.
	 *
	 * @since 3.7
	 * @deprecated 3.9.0
	 * @return void
	 */
	public function display_renewal_email_preview() {}

	/**
	 * Add {license_keys} Email Tag.
	 *
	 * @since 2.4
	 * @deprecated 3.8.12
	 * @access public
	 */
	public function add_email_tag() {}

	/**
	 * Get the license keys for a purchase.
	 *
	 * @since unknown
	 * @deprecated 3.8.12
	 * @param int $payment_id The order ID.
	 * @return string
	 */
	public function licenses_tag( $payment_id = 0 ) {
		$tags = new \EDD\SoftwareLicensing\Emails\Tags();

		return $tags->licenses_tag( $payment_id );
	}
}
