<?php

/**
 * Registers the Software Licensing license section under the extensions tab.
 *
 * @param array $sections The existing plugin sections.
 * @return array The modified plugin sections
 */
function edd_sl_register_license_section( $sections ) {
	$sections['software-licensing'] = __( 'Software Licensing', 'edd_sl' );

	return $sections;
}
add_filter( 'edd_settings_sections_extensions', 'edd_sl_register_license_section', 10, 1 );

/**
 * Registers the Software Licensing license options under the extensions tab.
 *
 * @since 1.0
 * @param array $settings The existing plugin settings.
 * @return array
 */
function edd_sl_license_settings( $settings ) {
	return array_merge( $settings, array( 'software-licensing' => EDD\SoftwareLicensing\Admin\Settings::get() ) );
}
add_filter( 'edd_settings_extensions', 'edd_sl_license_settings' );

/**
 * Processes the deletion of an existing renewal notice
 *
 * @since 3.0
 * @param array $data The post data.
 * @return void
 */
function edd_sl_process_delete_renewal_notice( $data ) {

	if ( ! is_admin() ) {
		return;
	}

	if ( ! current_user_can( 'manage_shop_settings' ) ) {
		wp_die( __( 'You do not have permission to add renewal notices', 'edd_sl' ), __( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
	}

	if ( ! wp_verify_nonce( $data['_wpnonce'] ) ) {
		wp_die( __( 'Nonce verification failed', 'edd_sl' ), __( 'Error', 'edd_sl' ), array( 'response' => 401 ) );
	}

	if ( empty( $data['notice-id'] ) && 0 !== (int) $data['notice-id'] ) {
		wp_die( __( 'No renewal notice ID was provided', 'edd_sl' ), __( 'Error', 'edd_sl' ), array( 'response' => 409 ) );
	}

	$deleted = false;
	if ( function_exists( 'edd_get_email_by' ) ) {
		$email_id = "license_{$data['notice-id']}";
		$email    = edd_get_email_by( 'email_id', $email_id );
		if ( $email ) {
			$deleted = edd_delete_email( $email->id );
		}
	}

	$notices = edd_sl_get_renewal_notices();
	// Ensure the old notice is deleted if it exists.
	if ( isset( $notices[ absint( $data['notice-id'] ) ] ) ) {
		unset( $notices[ absint( $data['notice-id'] ) ] );

		update_option( 'edd_sl_renewal_notices', $notices );
	}

	edd_redirect(
		edd_get_admin_url(
			array(
				'page' => 'edd-emails',
			)
		)
	);
}
add_action( 'edd_delete_renewal_notice', 'edd_sl_process_delete_renewal_notice' );

/**
 * Gets the default text for the renewal notices.
 *
 * @since 3.7
 * @return string
 */
function edd_sl_get_default_renewal_notice_message() {
	return 'Hello {name},

Your license key for {product_name} is about to expire.

If you wish to renew your license, simply click the link below and follow the instructions.

Your license expires on: {expiration}.

Your expiring license key is: {license_key}.

Renew now: {renewal_link}.';
}
