<?php

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class EDD_SL_Renewal_Notice_Logs extends WP_List_Table {

	private $per_page = 20;

	public function __construct() {

		//Set parent defaults
		parent::__construct(
			array(
				'singular' => 'renewal_notice',  //singular name of the listed records
				'plural'   => 'renewal_notices', //plural name of the listed records
				'ajax'     => false,              //does this table support ajax?
			)
		);
	}

	/**
	 * Setup columns
	 *
	 * @access      public
	 * @since       3.0
	 * @return      array
	 */
	public function get_columns() {

		return array(
			'cb'        => '<input type="checkbox"/>',
			'recipient' => __( 'Product - Recipient', 'edd_sl' ),
			'subject'   => __( 'Email Subject', 'edd_sl' ),
			'date'      => __( 'Date Sent', 'edd_sl' ),
		);
	}

	/**
	 * Output the checkbox column.
	 *
	 * @since       3.0
	 * @return      void
	 */
	protected function column_cb( $item ) {

		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			esc_attr( $this->_args['singular'] ),
			esc_attr( $item->ID )
		);
	}

	/**
	 * Output the recipient column
	 *
	 * @access      public
	 * @since       3.0
	 * @return      void
	 */
	protected function column_recipient( $item ) {
		if ( ! empty( $item->content ) ) {
			return $item->content;
		}
		$license = edd_software_licensing()->get_license( $item->license_id );
		if ( ! $license ) {
			return $item->id;
		}

		return $license ? $license->get_download()->get_name() . ' &mdash; ' . $license->customer->email : $item->ID;
	}

	/**
	 * Output the subject column
	 *
	 * @access      public
	 * @since       3.0
	 * @return      void
	 */
	protected function column_subject( $item ) {
		if ( ! empty( $item->subject ) ) {
			return $item->subject;
		}
		if ( ! metadata_exists( 'edd_logs_license', $item->id, 'renewal_notice_id' ) ) {
			return '';
		}
		$notice_id = get_metadata( 'edd_logs_license', $item->id, 'renewal_notice_id', true );
		if ( false === $notice_id ) {
			return '';
		}
		if ( function_exists( 'edd_get_email_by' ) ) {
			$notice = edd_get_email_by( 'email_id', $notice_id );
			if ( $notice ) {
				return $notice->subject;
			}
		}

		$notice = edd_sl_get_renewal_notice( $notice_id );

		return ! empty( $notice['subject'] ) ? $notice['subject'] : '';
	}

	/**
	 * Output the date column
	 *
	 * @access      public
	 * @since       3.0
	 * @return      void
	 */
	protected function column_date( $item ) {
		return $item->date_created;
	}

	/**
	 * Retrieve the current page number
	 *
	 * @access      public
	 * @since       3.0
	 * @return      int
	 */
	public function get_paged() {
		return isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
	}

	/**
	 * Outputs the log views
	 *
	 * @access public
	 * @since  3.0
	 * @return void
	 */
	public function bulk_actions( $which = '' ) {
		// These aren't really bulk actions but this outputs the markup in the right place
		edd_log_views();
	}

	/**
	 * Retrieve the current page number
	 *
	 * @access      public
	 * @since       3.0
	 * @return      int
	 */
	protected function count_total_items() {

		$args = array(
			'meta_query' => array(
				array(
					'key'     => 'renewal_notice_id',
					'compare' => 'EXISTS',
				),
			),
			'count'      => true,
		);

		$query = new EDD\SoftwareLicensing\Database\Queries\LogLicense( $args );

		return absint( $query->found_items );
	}

	/**
	 * Query database for license data and prepare it for the table
	 *
	 * @access      public
	 * @since       3.0
	 * @return      array
	 */
	public function logs_data() {
		$args = array(
			'number'     => 30,
			'meta_query' => array(
				array(
					'key'     => 'renewal_notice_id',
					'compare' => 'EXISTS',
				),
			),
		);

		$query = new EDD\SoftwareLicensing\Database\Queries\LogLicense();

		return $query->query( $args );
	}

	/**
	 * Sets up the list table items
	 *
	 * @access      public
	 * @since       3.0
	 * @return      void
	 */
	public function prepare_items() {

		$columns = $this->get_columns();

		$this->_column_headers = array( $columns, array(), array() );

		$this->items = $this->logs_data();

		$total_items = $this->count_total_items();

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $this->per_page,
				'total_pages' => ceil( $total_items / $this->per_page ),
			)
		);
	}
}
