<?php

namespace EDD\SoftwareLicensing\Upgrades;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

trait UpgradeNotification {

	/**
	 * Marks the migration as complete.
	 *
	 * @since 3.9.0
	 *
	 * @param bool $add_notification If we should add a notification that the migration is complete.
	 *
	 * @return void
	 */
	protected function mark_complete( $add_notification = true ) {
		// Set the upgrade as complete.
		edd_set_upgrade_complete( $this->upgrade_name );

		if ( false === $add_notification ) {
			return;
		}

		$initial_notification = $this->get_initial_notification();
		if ( ! empty( $initial_notification ) ) {
			EDD()->notifications->update( $initial_notification->id, array( 'dismissed' => 1 ) );
		}

		$parameters = $this->get_complete_notification_parameters();
		$parameters = wp_parse_args(
			$parameters,
			array(
				'buttons'    => '',
				'conditions' => '',
				'type'       => 'success',
			)
		);
		EDD()->notifications->maybe_add_local_notification( $parameters );
	}

	/**
	 * Gets the initial notification about the migration.
	 *
	 * @since 3.9.0
	 *
	 * @return object
	 */
	private function get_initial_notification() {
		$parameters = $this->get_initial_notification_parameters();

		return EDD()->notifications->get_item_by( 'remote_id', $parameters['remote_id'] );
	}

	/**
	 * Adds or updates the initial notification about the migration.
	 *
	 * @since 3.9.0
	 *
	 * @return void
	 */
	protected function add_or_update_initial_notification() {
		$initial_notification = $this->get_initial_notification();
		$percent_complete     = $this->get_percentage_complete();
		$parameters           = $this->get_initial_notification_parameters();
		$parameters           = wp_parse_args(
			$parameters,
			array(
				'buttons'    => '',
				'conditions' => '',
				'type'       => 'info',
			)
		);

		$parameters['title'] .= sprintf( ' ( %d%% )', $percent_complete );

		if ( ! empty( $initial_notification ) ) {
			$date = new \EDD\Utils\Date();
			$date->setTimestamp( time() )->setTimezone( new \DateTimeZone( 'UTC' ) );

			// Update the notification.
			EDD()->notifications->update(
				$initial_notification->id,
				array(
					'title'        => $parameters['title'],
					'date_created' => $date->format( 'mysql' ),
				)
			);
		} else {
			// Add the notification.
			EDD()->notifications->maybe_add_local_notification( $parameters );
		}
	}
}
