<?php

namespace EDD\SoftwareLicensing\Upgrades;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\SoftwareLicensing\Database\Queries\LogLicense as Query;
use EDD\SoftwareLicensing\Downloads\LicensedProduct;

class LicenseLogs {
	use UpgradeNotification;

	private $total_count_option = 'edd_sl_license_logs_total_count';

	/**
	 * The name of the upgrade.
	 *
	 * @var string
	 */
	protected $upgrade_name = 'sl_license_logs';

	/**
	 * LicenseLogs constructor.
	 */
	public function __construct() {
		// If the upgrade has already been run, don't hook anything.
		if ( edd_has_upgrade_completed( $this->upgrade_name ) ) {
			return;
		}

		add_action( 'sl_migrate_license_logs', array( $this, 'process_step' ) );
		if ( ! wp_next_scheduled( 'sl_migrate_license_logs' ) ) {
			add_action( 'shutdown', array( $this, 'maybe_schedule_background_update' ), 99 );
		}
	}

	/**
	 * Maybe schedule the background update.
	 *
	 * @since 3.9.0
	 */
	public function maybe_schedule_background_update(): void {
		global $wpdb;
		// count any posts that have a post type of 'edd_log' and a post status of 'publish'
		$count = $wpdb->get_var( "SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'edd_license_log' AND post_status = 'publish'" );

		if ( empty( $count ) ) {
			$this->mark_complete( false );
			return;
		}

		// Only update the total count option if it doesn't exist. This prevents it from being overridden in some edge cases.
		if ( empty( get_option( $this->total_count_option ) ) ) {
			// Set the total amount in a transient so we can use it later.
			update_option( $this->total_count_option, $count, false );
		}

		$this->add_or_update_initial_notification();

		// ...And schedule a single event a minute from now to start the processing of this data.
		wp_schedule_single_event( time() + MINUTE_IN_SECONDS, 'sl_migrate_license_logs' );
	}

	/**
	 * Process the step.
	 *
	 * @since 3.9.0
	 */
	public function process_step(): void {
		if ( ! edd_doing_cron() ) {
			return;
		}

		global $wpdb;
		$rows = $wpdb->get_results( "SELECT * FROM $wpdb->posts WHERE post_type = 'edd_license_log' AND post_status = 'publish' LIMIT 100" );

		edd_debug_log( 'Found ' . count( $rows ) . ' license logs to process.' );

		if ( empty( $rows ) ) {
			$this->mark_complete();
			return;
		}

		$query = new Query();
		foreach ( $rows as $row ) {
			edd_debug_log( 'Processing log ID: ' . $row->ID );
			$args            = array(
				'subject'      => $row->post_title,
				'content'      => $row->post_content,
				'date_created' => $row->post_date_gmt,
			);
			$meta            = get_post_meta( $row->ID );
			if ( array_key_exists( '_edd_sl_log_license_id', $meta ) ) {
				edd_debug_log( 'Found license ID: ' . $meta['_edd_sl_log_license_id'][0] );
				$args['license_id'] = absint( $meta['_edd_sl_log_license_id'][0] );
				unset( $meta['_edd_sl_log_license_id'] );
			}
			$new_meta = array();
			if ( ! empty( $meta ) ) {
				edd_debug_log( 'Found meta: ' . print_r( $meta, true ) );
				foreach ( $meta as $key => $value ) {
					$key              = str_replace( '_edd_sl_log_', '', $key );
					$key              = str_replace( '_edd_sl_', '', $key );
					$new_meta[ $key ] = maybe_unserialize( $value[0] );
				}
			}
			if ( false !== strpos( $row->post_title, 'Renewed via Subscription' ) ) {
				edd_debug_log( 'Found subscription message: ' . $row->post_content );
				$new_meta['subscription_id'] = absint( $row->post_content );
				unset( $args['content'] );
			}
			// If this is a renewal notice, we need to get the license and the subject. We are not saving the original post title.
			if ( isset( $new_meta['renewal_notice_id'] ) ) {
				edd_debug_log( 'Found renewal notice ID: ' . $new_meta['renewal_notice_id'] );
				// Get the download ID from the license row.
				$download_id = $wpdb->get_var( "SELECT download_id FROM $wpdb->edd_licenses WHERE id = " . $args['license_id'] );
				$download    = new LicensedProduct( $download_id );

				// Get the customer ID from the license row.
				$customer_id = $wpdb->get_var( "SELECT customer_id FROM $wpdb->edd_licenses WHERE id = " . $args['license_id'] );
				$customer    = edd_get_customer( $customer_id );
				if ( $download && $customer ) {
					$args['content'] = $download->get_name() . ' &mdash; ' . $customer->email;
				}
				$notice          = edd_sl_get_renewal_notice( $new_meta['renewal_notice_id'] );
				$args['subject'] = $notice['subject'];
			}

			edd_debug_log( 'Adding log: ' . $args['subject'] );
			$log_id = $query->add_item( $args );

			edd_debug_log( 'Log ID: ' . $log_id );

			if ( ! empty( $new_meta ) ) {
				foreach ( $new_meta as $key => $value ) {
					add_metadata( 'edd_logs_license', $log_id, $key, $value, true );
				}
			}

			edd_debug_log( 'Successfully migrated legacy log ID: ' . $row->ID );
			wp_delete_post( $row->ID, true );
		}

		$this->add_or_update_initial_notification();

		wp_schedule_single_event( time() + MINUTE_IN_SECONDS, 'sl_migrate_license_logs' );
	}

	/**
	 * Get the percentage complete.
	 *
	 * @since 3.9.0
	 * @return int
	 */
	protected function get_percentage_complete(): int {
		static $percent_complete;

		if ( ! is_null( $percent_complete ) ) {
			return $percent_complete;
		}

		global $wpdb;

		// Get the total amount of rows remaining.
		$total_rows_remaining = $wpdb->get_var( "SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'edd_license_log' AND post_status = 'publish'" );

		// Get the total we started with.
		$total_rows_start = get_option( $this->total_count_option );

		// Format a % complete without decimals.
		$percent_complete = number_format( ( ( $total_rows_start - $total_rows_remaining ) / $total_rows_start ) * 100, 0 );

		// Just in case we end up over 100%, somehow...make it 99%;
		if ( $percent_complete > 99 ) {
			$percent_complete = 99;
		}

		return $percent_complete;
	}

	/**
	 * Get the initial notification parameters.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	protected function get_initial_notification_parameters() {
		return array(
			'remote_id' => 'sl_logs_start',
			'title'     => __( 'Starting License Logs Upgrade', 'edd_sl' ),
			'content'   => __( 'We are upgrading the license logs for your site. This process may take some time to complete.', 'edd_sl' ),
		);
	}

	/**
	 * Get the complete notification parameters.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	protected function get_complete_notification_parameters() {
		return array(
			'remote_id' => 'sl_logs_complete',
			'title'     => __( 'License Logs Upgrade Complete', 'edd_sl' ),
			'content'   => __( 'The license logs upgrade is complete!', 'edd_sl' ),
		);
	}
}
