<?php
/**
 * GetVersion remote request.
 *
 * @package   EDD\SoftwareLicensing\RemoteRequests\Requests
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\RemoteRequests\Requests;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * GetVersion class.
 */
class GetVersion extends Request {

	/**
	 * The URL of the item.
	 *
	 * @var string
	 */
	private $url = false;

	/**
	 * The license key.
	 *
	 * @var string
	 */
	private $license = false;

	/**
	 * The slug of the item.
	 *
	 * @var string
	 */
	private $slug = false;

	/**
	 * The item ID.
	 *
	 * @var int
	 */
	private $item_id = false;

	/**
	 * The item name.
	 *
	 * @var string
	 */
	private $item_name = false;

	/**
	 * Whether the item is a beta version.
	 *
	 * @var bool
	 */
	private $beta = false;

	/**
	 * The data received from the item to check.
	 *
	 * @var array
	 */
	private $data = array();

	/**
	 * The download object.
	 *
	 * @var \EDD\SoftwareLicensing\Downloads\LicensedProduct
	 */
	private $download = false;

	/**
	 * Process both single and multiple licence checks.
	 *
	 * @since 3.9.0
	 * @param array $data The information of the item to check.
	 * @return array|void
	 */
	public function handle( $data ) {
		$should_return = apply_filters( 'edd_remote_latest_version_return', false );
		$this->parse_data( $data );

		if ( ! isset( $data['products'] ) ) {
			$response = $this->get_single_latest_version_remote();

			if ( $should_return ) {
				return $response;
			}

			$this->send_response( $response );
			return;
		}

		if ( empty( $data['products'] ) ) {
			$response['msg'] = __( 'No item provided', 'edd_sl' );

			if ( $should_return ) {
				return $response;
			}

			$this->send_response( $response );
			return;
		}

		$result = array();
		foreach ( $data['products'] as $slug => $single_data ) {
			$this->reset_data();
			$this->parse_data( $single_data );
			$result[ $slug ] = $this->get_single_latest_version_remote();
		}

		if ( $should_return ) {
			return $result;
		}

		$this->send_response( $result );
	}

	/**
	 * Parse the data received from the item to check.
	 *
	 * @since 3.9.1
	 * @param array $data The data received from the item to check.
	 */
	private function parse_data( $data ) {
		if ( isset( $data['url'] ) ) {
			$this->url = sanitize_text_field( urldecode( $data['url'] ) );
		}
		if ( isset( $data['license'] ) ) {
			$this->license = sanitize_text_field( urldecode( $data['license'] ) );
		}
		if ( isset( $data['slug'] ) ) {
			$this->slug = sanitize_text_field( urldecode( $data['slug'] ) );
		}
		if ( isset( $data['item_id'] ) ) {
			$this->item_id = absint( $data['item_id'] );
		}
		if ( isset( $data['item_name'] ) ) {
			$this->item_name = sanitize_text_field( rawurldecode( $data['item_name'] ) );
		}
		if ( isset( $data['beta'] ) ) {
			$this->beta = (bool) $data['beta'];
		}
		if ( empty( $this->item_name ) && empty( $this->item_id ) && isset( $data['name'] ) ) {
			$this->item_name = sanitize_text_field( rawurldecode( $data['name'] ) );
		}

		if ( empty( $this->item_id ) ) {
			$check_by_name_first = apply_filters( 'edd_sl_force_check_by_name', false );
			if ( empty( $this->license ) || $check_by_name_first ) {
				$this->item_id = edd_software_licensing()->get_download_id_by_name( $this->item_name );
			} else {
				$this->item_id = edd_software_licensing()->get_download_id_by_license( $this->license );
			}
		}

		$this->data = $data;
	}

	/**
	 * Reset the data.
	 *
	 * @since 3.9.1
	 */
	private function reset_data() {
		$this->data      = array();
		$this->download  = false;
		$this->item_id   = false;
		$this->item_name = false;
		$this->license   = false;
		$this->slug      = false;
		$this->beta      = false;
		$this->url       = false;
	}

	/**
	 * Process an individual license check.
	 *
	 * @since 3.7.1
	 * @return array The item information or an error message.
	 */
	private function get_single_latest_version_remote() {

		/**
		 * Filters the license check response.
		 *
		 * @since 1.2
		 * @since 3.7.2 Added the `$data` parameter.
		 *
		 * @param array                                            $response      Response.
		 * @param \EDD\SoftwareLicensing\Downloads\LicensedProduct $download      Download object.
		 * @param bool                                             $download_beta Whether there is a beta download available.
		 * @param array                                            $data          Request data sent to check the item.
		 */
		$response = apply_filters(
			'edd_sl_license_response',
			wp_parse_args( $this->get_response(), $this->get_defaults() ),
			$this->get_download(),
			$this->beta,
			$this->data
		);

		// Logs the request. This will create a new activation if one doesn't exist, if the setting to log all activations is enabled.
		$log_all_requests = (bool) edd_get_option( 'edd_sl_log_all_requests', false );
		$this->log_remote_request( $this->license, $this->url, $this->data, $log_all_requests );

		/**
		 * Encode any emoji in the name and sections.
		 *
		 * @since 3.6.5
		 * @see https://github.com/easydigitaldownloads/EDD-Software-Licensing/issues/1313
		 */
		$response['name'] = wp_encode_emoji( $response['name'] );

		$sections = maybe_unserialize( $response['sections'] );
		if ( is_array( $sections ) ) {
			$response['sections'] = serialize( array_map( 'wp_encode_emoji', $sections ) );
		} else {
			$response['sections'] = wp_encode_emoji( $sections );
		}

		return $response;
	}

	/**
	 * Get the response for the remote request.
	 *
	 * @since 3.9.1
	 * @return array The response.
	 */
	private function get_response() {
		$response = array();

		$bypass_name_check = defined( 'EDD_BYPASS_NAME_CHECK' ) && EDD_BYPASS_NAME_CHECK;
		if ( empty( $this->item_id ) && empty( $this->item_name ) && ! $bypass_name_check ) {
			$response['msg'] = __( 'No item provided', 'edd_sl' );

			return $response;
		}

		if ( empty( $this->item_id ) ) {
			$response['msg'] = __( 'No item provided', 'edd_sl' );

			return $response;
		}

		$download = $this->get_download();

		if ( ! $download || empty( $download->ID ) ) {
			if ( empty( $this->license ) || ! empty( apply_filters( 'edd_sl_force_check_by_name', false ) ) ) {
				/* translators: the singular download label */
				$response['msg'] = sprintf( __( 'Item name provided does not match a valid %s', 'edd_sl' ), edd_get_label_singular() );
			} else {
				/* translators: the singular download label */
				$response['msg'] = sprintf( __( 'License key provided does not match a valid %s', 'edd_sl' ), edd_get_label_singular() );
			}

			return $response;
		}

		$message               = '';
		$is_valid_for_download = edd_software_licensing()->is_download_id_valid_for_license( $download->ID, $this->license );
		if ( empty( $this->license ) ) {
			$message = __( 'No license key has been provided.', 'edd_sl' );
		} elseif ( ! $bypass_name_check && ( ! $is_valid_for_download || ( ! empty( $this->item_name ) && ! edd_software_licensing()->check_item_name( $download->ID, $this->item_name, $this->license ) ) ) ) {
			$download_name = ! empty( $this->item_name ) ? $this->item_name : $download->get_name();
			/* translators: the download name */
			$response['msg'] = sprintf( __( 'License key is not valid for %s', 'edd_sl' ), $download_name );

			return $response;
		}

		$stable_version = edd_software_licensing()->get_latest_version( $this->item_id );
		$new_version    = $stable_version;
		$description    = ! empty( $download->post_excerpt ) ? $download->post_excerpt : $download->post_content;
		$description    = strip_shortcodes( $description );
		$changelog      = $download->get_changelog( true );

		$download_beta = false;
		if ( $this->beta && $download->has_beta() ) {
			$version_beta = edd_software_licensing()->get_beta_download_version( $this->item_id );
			if ( version_compare( $version_beta, $stable_version, '>' ) ) {
				$changelog     = $download->get_beta_changelog();
				$new_version   = $version_beta;
				$download_beta = true;
			}
		}

		$new_version = $this->maybe_get_new_version_from_requirements( $download, $new_version );
		$package     = edd_software_licensing()->get_encoded_download_package_url( $this->item_id, $this->license, $this->url, $download_beta );
		$response    = array(
			'new_version'    => $new_version,
			'stable_version' => $stable_version,
			'name'           => $download->post_title,
			'slug'           => ! empty( $this->slug ) ? $this->slug : $download->post_name,
			'url'            => esc_url( add_query_arg( 'changelog', '1', get_permalink( $this->item_id ) ) ),
			'last_updated'   => $download->post_modified_gmt,
			'homepage'       => get_permalink( $this->item_id ),
			'package'        => $package,
			'download_link'  => $package,
			'sections'       => serialize(
				array(
					'description' => wpautop( strip_tags( $description, '<p><li><ul><ol><strong><a><em><span><br>' ) ),
					'changelog'   => wpautop( strip_tags( stripslashes( $changelog ), '<p><li><ul><ol><strong><a><em><span><br>' ) ),
				)
			),
			'banners'        => serialize(
				array(
					'high' => get_post_meta( $this->item_id, '_edd_readme_plugin_banner_high', true ),
					'low'  => get_post_meta( $this->item_id, '_edd_readme_plugin_banner_low', true ),
				)
			),
			'icons'          => array(),
		);

		if ( $message ) {
			$response['msg'] = $message;
		}

		if ( has_post_thumbnail( $download->ID ) ) {
			$thumb_128 = get_the_post_thumbnail_url( $download->ID, 'sl-small' );
			if ( ! empty( $thumb_128 ) ) {
				$response['icons']['1x'] = $thumb_128;
			}

			$thumb_256 = get_the_post_thumbnail_url( $download->ID, 'sl-large' );
			if ( ! empty( $thumb_256 ) ) {
				$response['icons']['2x'] = $thumb_256;
			}
		}

		$response['icons'] = serialize( $response['icons'] );

		return $response;
	}

	/**
	 * Maybe get the new version from the requirements.
	 *
	 * @since 3.9.1
	 *
	 * @param \EDD\SoftwareLicensing\Downloads\LicensedProduct $download The download object.
	 * @return bool|string The new version or false if no new version is found.
	 */
	private function maybe_get_new_version_from_requirements( $download, $new_version ) {
		$download_requirements = $download->get_requirements();
		if ( empty( $download_requirements ) ) {
			return $new_version;
		}

		$requirements_data = $this->get_requirements_data( $download_requirements );
		if ( empty( $requirements_data ) ) {
			return $new_version;
		}

		$requirements = new \EDD\SoftwareLicensing\Requirements\Requirements( $requirements_data );
		if ( ! $requirements->met() && isset( $this->data['version'] ) ) {
			return $this->data['version'];
		}

		return $new_version;
	}

	/**
	 * Get the download object.
	 *
	 * @since 3.9.1
	 * @return \EDD\SoftwareLicensing\Downloads\LicensedProduct The download object.
	 */
	private function get_download() {
		if ( $this->download ) {
			return $this->download;
		}

		$this->download = new \EDD\SoftwareLicensing\Downloads\LicensedProduct( $this->item_id );

		return $this->download;
	}

	/**
	 * Get the requirements data.
	 *
	 * @since 3.9.1
	 * @param array $download_requirements The download requirements.
	 * @return array The requirements data.
	 */
	private function get_requirements_data( $download_requirements ) {
		$requirements_data = array();

		foreach ( $download_requirements as $platform => $min_version ) {
			$platform         = sanitize_text_field( $platform );
			$platform_version = "{$platform}_version";
			if ( ! empty( $this->data[ $platform_version ] ) ) {
				$requirements_data[ $platform ] = array(
					'minimum' => sanitize_text_field( $min_version ),
					'current' => sanitize_text_field( $this->data[ $platform_version ] ),
					'local'   => false,
				);
			}
		}

		/**
		 * Allow filtering the update requirements data, used by the EDD\SoftwareLicensing\Requirements\Requirements class.
		 *
		 * @since 3.8
		 *
		 * @param array The update requirements matched against the user's current system.
		 * @param array The requirements assigned to the download item.
		 * @param array The $data received from the item to check.
		 */
		return apply_filters( 'edd_sl_download_requirements_data', $requirements_data, $download_requirements, $this->data );
	}

	/**
	 * Get the defaults.
	 *
	 * @since 3.9.1
	 * @return array The defaults.
	 */
	private function get_defaults() {
		return array(
			'name'           => '',
			'new_version'    => '',
			'stable_version' => '',
			'sections'       => '',
			'license_check'  => '',
			'msg'            => '',
			'homepage'       => '',
			'package'        => '',
			'icons'          => array(),
			'banners'        => array(),

		);
	}
}
