<?php
/**
 * Deactivate remote license request.
 *
 * @package   EDD\SoftwareLicensing\RemoteRequests\Requests
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\RemoteRequests\Requests;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Deactivate class.
 */
class Deactivate extends Request {

	/**
	 * Handle the request.
	 *
	 * @since 3.9.0
	 * @param array $data
	 * @return void
	 */
	public function handle( $data ) {

		$item_id     = ! empty( $data['item_id'] ) ? absint( $data['item_id'] ) : false;
		$item_name   = ! empty( $data['item_name'] ) ? rawurldecode( $data['item_name'] ) : get_the_title( $item_id );
		$license_key = urldecode( $data['license'] );

		$args   = array(
			'item_id'     => $item_id,
			'item_name'   => $item_name,
			'key'         => $license_key,
			'url'         => isset( $data['url'] ) ? urldecode( $data['url'] ) : '',
			'environment' => empty( $data['environment'] ) ? 'production' : $data['environment'],
		);
		$result = $this->deactivate( $args );

		$status = 'failed';
		if ( $result ) {
			$status = 'deactivated';
		}

		$license  = edd_software_licensing()->get_license( $license_key, true );
		$response = array(
			'success'   => (bool) $result,
			'license'   => $status,
			'item_id'   => $item_id,
			'item_name' => $item_name,
			'checksum'  => $this->get_request_checksum( $args ),
		);
		if ( false !== $license ) {
			if ( $result ) {
				$this->log_remote_request( $license->id, $args['url'] );
			}
			$response['expires']        = $license->expiration;
			$response['payment_id']     = $license->payment_id;
			$response['customer_name']  = $license->customer->name;
			$response['customer_email'] = $license->customer->email;
			$response['price_id']       = $license->price_id;
		}

		$license_id = ! empty( $license ) ? $license->ID : 0;
		$response   = apply_filters( 'edd_remote_license_deactivation_response', $response, $args, $license_id );

		$this->send_response( $response );
	}

	/**
	 * Deactivate the license.
	 *
	 * @since 3.9.0
	 * @param array $args
	 * @return bool
	 */
	private function deactivate( $args ) {
		$deactivate = new \EDD\SoftwareLicensing\Licenses\Actions\Deactivate();

		return $deactivate->deactivate_license( $args );
	}
}
