<?php
/**
 * Check remote license request.
 *
 * @package   EDD\SoftwareLicensing\RemoteRequests\Requests
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\RemoteRequests\Requests;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Check class.
 */
class Check extends Request {

	/**
	 * Handle the request.
	 *
	 * @since 3.9.0
	 * @param array $data
	 * @return array|void
	 */
	public function handle( $data ) {

		$item_id   = ! empty( $data['item_id'] ) ? absint( $data['item_id'] ) : false;
		$item_name = ! empty( $data['item_name'] ) ? rawurldecode( $data['item_name'] ) : get_the_title( $item_id );
		$license   = isset( $data['license'] ) ? urldecode( $data['license'] ) : false;
		$url       = isset( $data['url'] ) ? urldecode( $data['url'] ) : '';

		$args = array(
			'item_id'   => $item_id,
			'item_name' => $item_name,
			'key'       => $license,
			'url'       => $url,
		);

		$result   = $this->check( $args );
		$response = array(
			'success'   => (bool) ( 'invalid' !== $result ),
			'license'   => $result,
			'item_id'   => $item_id,
			'item_name' => $item_name,
			'checksum'  => $this->get_request_checksum( $args ),
		);
		if ( 'invalid' !== $result ) {
			$license = edd_software_licensing()->get_license( $license, true );
			if ( false !== $license ) {
				$this->log_remote_request( $license->id, $args['url'], array(), false );
			}

			$response['expires']          = is_numeric( $license->expiration ) ? date( 'Y-m-d H:i:s', $license->expiration ) : $license->expiration;
			$response['payment_id']       = $license->payment_id;
			$response['customer_name']    = $license->customer->name;
			$response['customer_email']   = $license->customer->email;
			$response['license_limit']    = $license->activation_limit;
			$response['site_count']       = $license->activation_count;
			$response['activations_left'] = $license->activation_limit > 0 ? $license->activation_limit - $license->activation_count : 'unlimited';
			$response['price_id']         = $license->price_id;
		}

		$license_id = ! empty( $license->ID ) ? $license->ID : false;
		$response   = apply_filters( 'edd_remote_license_check_response', $response, $args, $license_id );

		if ( apply_filters( 'edd_remote_license_check_return', false ) ) {
			return $response;
		}

		$this->send_response( $response );
	}

	/**
	 * Check the license.
	 *
	 * @param array $args
	 * @return string
	 */
	private function check( $args ) {
		$check = new \EDD\SoftwareLicensing\Licenses\Actions\Check();

		return $check->check_license( $args );
	}
}
