<?php
/**
 * Software Licensing Upgrades.
 *
 * @package EDD\SoftwareLicensing\Licenses\Upgrades
 * @since   3.9.0
 */

namespace EDD\SoftwareLicensing\Licenses;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Upgrades class.
 */
class Upgrades {

	/**
	 * Check if a license has available upgrades.
	 *
	 * @since 3.9.0
	 * @param int $license_id License ID.
	 * @return bool
	 */
	public static function license_has_upgrades( $license_id ) {
		if ( empty( $license_id ) ) {
			return false;
		}

		$license = edd_software_licensing()->get_license( $license_id );
		if ( false === $license || in_array( $license->status, array( 'disabled', 'revoked' ), true ) ) {
			return false;
		}

		// If this is a child license, upgrades are not available.
		if ( ! empty( $license->parent ) ) {
			return false;
		}

		$upgrade_paths = edd_sl_get_upgrade_paths( $license->download_id );
		if ( ! is_array( $upgrade_paths ) ) {
			return false;
		}

		$order_id = edd_software_licensing()->get_payment_id( $license_id );
		$order    = edd_get_order( $order_id );

		if ( ! $order || 'complete' !== $order->status ) {
			return false;
		}

		$download_id = (int) $license->download_id;

		foreach ( $upgrade_paths as $path ) {

			if ( edd_has_variable_prices( $download_id ) ) {

				// If there is a different product in the upgrade paths, upgrade is available.
				if ( (int) $path['download_id'] !== $download_id ) {
					return true;
				}

				// If same download but with a more expensive price ID is in upgrade paths, upgrade is available.
				if ( (int) $path['price_id'] !== (int) $license->price_id && edd_get_price_option_amount( $path['download_id'], $path['price_id'] ) > edd_get_price_option_amount( $download_id, $license->price_id ) ) {
					return true;
				}
			} elseif ( (int) $path['download_id'] !== $download_id ) {
				return true;
			}
		}

		return false;
	}
}
