<?php
/**
 * Software Licensing License Generator Trait
 *
 * @package   EDD\SoftwareLicensing\Licenses\Traits\Generator
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Licenses\Traits;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Generator trait.
 *
 * @since 3.9.0
 */
trait Generator {

	/**
	 * Generate license keys for a purchase
	 *
	 * Generates ( if needed ) a license key for the buyer at time of purchase
	 * This key will be used to activate all products for this purchase
	 *
	 * @access      public
	 * @since       1.5
	 * @uses EDD_Software_Licensing::generate_new_license
	 *
	 * @param int    $download_id The download ID for the license.
	 * @param int    $payment_id  The associated payment ID.
	 * @param string $type        The type of product (default/bundle).
	 * @param array  $cart_item   The information about the item in the cart.
	 * @param mixed  $cart_index  The position of the item in the cart.
	 *
	 * @return      mixed
	 */
	public function generate_license( $download_id = 0, $payment_id = 0, $type = 'default', $cart_item = array(), $cart_index = 0 ) {

		$args = array(
			'download_id' => $download_id,
			'payment_id'  => $payment_id,
			'type'        => $type,
			'cart_item'   => $cart_item,
			'cart_index'  => $cart_index,
		);

		return $this->generate_new_license( $args );
	}

	/**
	 * Generates a new license.
	 *
	 * @since 3.7
	 *
	 * @param array $args {
	 *     @type int     $download_id The download ID for the license.
	 *     @type int     $payment_id  The associated payment ID.
	 *     @type string  $type        The type of product (default/bundle).
	 *     @type array   $cart_item   The information about the item in the cart.
	 *     @type mixed   $cart_index  The position of the item in the cart.
	 *     @type boolean $retroactive Whether the license is being generated retroactively (default false).
	 *                                If true, licenses will be generated for upgrade payment, rather than the original.
	 * }
	 * @return array       An array of newly generated license keys.
	 */
	public function generate_new_license( $args = array() ) {

		$args = wp_parse_args(
			$args,
			array(
				'download_id' => 0,
				'payment_id'  => 0,
				'type'        => 'default',
				'cart_item'   => array(),
				'cart_index'  => 0,
				'retroactive' => false,
			)
		);

		$keys = array();

		// Bail if the download ID or payment ID is missing.
		if ( empty( $args['download_id'] ) || empty( $args['payment_id'] ) ) {
			return $keys;
		}

		$cart_item = $args['cart_item'];
		$price_id  = null;
		if ( $cart_item instanceof \EDD\Orders\Order_Item ) {
			$is_renewal = edd_get_order_item_meta( $cart_item->id, '_option_is_renewal', true );
			$is_upgrade = edd_get_order_item_meta( $cart_item->id, '_option_is_upgrade', true );
			$price_id   = $cart_item->price_id;
		} else {
			$is_renewal = ! empty( $cart_item['item_number']['options']['is_renewal'] );
			$is_upgrade = ! empty( $cart_item['item_number']['options']['is_upgrade'] );
			if ( isset( $args['cart_item']['item_number']['options']['price_id'] ) ) {
				$price_id = $args['cart_item']['item_number']['options']['price_id'];
			}
		}

		// Bail if this cart item is for a renewal.
		if ( $is_renewal ) {
			return $keys;
		}

		// Bail if the license is not being generated retroactively and is an upgrade.
		if ( empty( $args['retroactive'] ) && $is_upgrade ) {
			return $keys;
		}

		// If the related payment was upgraded, we should not generate a key.
		$upgraded = edd_get_order_meta( $args['payment_id'], '_edd_sl_upgraded_to_payment_id', true );
		if ( $upgraded ) {
			return $keys;
		}

		$options = array();
		/**
		 * Count the number of licenses already generated for this payment/download/cart index combination.
		 *
		 * @since 3.8.6 (when the filter was added)
		 */
		if ( apply_filters( 'edd_sl_check_existing_licenses', true ) ) {
			$args['count']                     = true;
			$query                             = new \EDD\SoftwareLicensing\Database\Queries\License( $args );
			$options['existing_license_count'] = $query->found_items;
		}

		$purchased_download = new \EDD_SL_Download( $args['download_id'] );
		if ( ! $purchased_download->is_bundled_download() && ! $purchased_download->licensing_enabled() ) {
			return $keys;
		}

		$license = new \EDD_SL_License();
		$license->create( $args['download_id'], $args['payment_id'], $price_id, $args['cart_index'], $options );

		if ( ! empty( $license->ID ) ) {
			$keys[] = $license->ID;

			$child_licenses = $license->get_child_licenses();
			if ( ! empty( $child_licenses ) ) {
				$child_ids = wp_list_pluck( $child_licenses, 'ID' );
				$keys      = array_merge( $keys, $child_ids );
			}
		}

		return $keys;
	}

	/**
	 * Generate a license key.
	 *
	 * @param int   $license_id
	 * @param int   $download_id
	 * @param int   $payment_id
	 * @param mixed $cart_index
	 * @param int   $timestamp  A numeric timestamp in order to attempt to 'salt' keys so they can be regenerated.
	 *
	 * @return string
	 */
	public function generate_license_key( $license_id = 0, $download_id = 0, $payment_id = 0, $cart_index = 0, $timestamp = 0 ) {
		$timestamp = is_numeric( $timestamp ) && ! empty( $timestamp ) ? absint( $timestamp ) : time();
		$key       = md5( $license_id . $download_id . $payment_id . $cart_index . $timestamp );

		return apply_filters( 'edd_sl_generate_license_key', $key, $license_id, $download_id, $payment_id, $cart_index );
	}
}
