<?php
/**
 * Deactivate License
 *
 * @package   EDD\SoftwareLicensing\Licenses\Actions
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Licenses\Actions;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Class Deactivate
 *
 * @since 3.9.0
 */
class Deactivate extends Action {

	/**
	 * Deactivate a license.
	 *
	 * @param array $args The arguments.
	 * @return bool
	 */
	public function deactivate_license( $args ) {

		$args = wp_parse_args( $args, $this->get_defaults() );
		$args = apply_filters( 'edd_sl_pre_deactivate_license_args', $args );

		// Remove any leading or trailing whitespace from the values in the arguments. See GH#1452
		$args = array_map( 'trim', $args );

		$license = edd_software_licensing()->get_license( $args['key'], true );

		if ( false === $license ) {
			return false;
		}

		$item_name = html_entity_decode( $args['item_name'] );

		if ( empty( $args['url'] ) && ! edd_software_licensing()->force_increase() ) {
			$args['url'] = \EDD\SoftwareLicensing\Utils\URL::get_url_from_user_agent();
		}

		$bypass_local = edd_get_option( 'edd_sl_bypass_local_hosts', false );
		$is_local_url = empty( $bypass_local ) ? false : edd_software_licensing()->is_local_url( $args['url'], $args['environment'] );

		do_action( 'edd_sl_pre_deactivate_license', $license->ID, $license->download_id );

		// make sure license is active
		if ( 'active' !== $license->status && ! $bypass_local ) {
			return false;
		}

		$allow_bundle_activation = apply_filters( 'edd_sl_allow_bundle_activation', false, $license );

		// Trying to deactivate bundle license
		if ( ! $allow_bundle_activation && $license->get_download()->is_bundled_download() ) {
			return false;
		}

		// don't deactivate if expired
		if ( ! $license->is_lifetime && $license->expiration < $args['expiration'] ) {
			return false; // this license has expired
		}

		if ( $args['key'] !== $license->key ) {
			return false; // keys don't match
		}

		if ( ! empty( $args['item_id'] ) ) {
			if ( ! edd_software_licensing()->is_download_id_valid_for_license( $license->download_id, $args['key'] ) ) {
				return false;
			}
		} elseif ( ( ! defined( 'EDD_BYPASS_NAME_CHECK' ) || ! EDD_BYPASS_NAME_CHECK ) && ! edd_software_licensing()->check_item_name( $license->download_id, $item_name, $license ) ) {
			return false; // Item names don't match
		}

		// deactivate the site for the license
		$removed = $license->remove_site( trailingslashit( $args['url'] ) );

		if ( ! $is_local_url ) {
			if ( apply_filters( 'edd_sl_log_license_deactivation', true ) ) {
				edd_software_licensing()->log_license_deactivation( $license->id, $_SERVER );
			}

			do_action( 'edd_sl_deactivate_license', $license->ID, $license->download_id );
		}

		return $removed; // license has been deactivated
	}
}
