<?php
/**
 * Activate a license
 *
 * @package   EDD\SoftwareLicensing\Licenses\Actions
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Licenses\Actions;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Activate class.
 */
class Activate extends Action {

	/**
	 * Activate a license.
	 *
	 * @param array $args The arguments.
	 * @return array
	 */
	public function activate_license( $args ) {

		$args = wp_parse_args( $args, $this->get_defaults() );
		$args = apply_filters( 'edd_sl_pre_activate_license_args', $args );

		// Remove any leading or trailing whitespace from the values in the arguments. See GH#1452.
		$args = array_map( 'trim', $args );

		$license = edd_software_licensing()->get_license( $args['key'], true );

		if ( empty( $args['url'] ) && ! edd_software_licensing()->force_increase() ) {
			$args['url'] = \EDD\SoftwareLicensing\Utils\URL::get_url_from_user_agent();
		}

		$license_id  = false !== $license ? $license->ID : 0;
		$download_id = false !== $license ? $license->download_id : 0;
		do_action( 'edd_sl_pre_activate_license', $license_id, $download_id );

		/**
		 * Allow developers to filter the activation result before it is returned.
		 *
		 * @param array $result   The activation result.
		 * @param object $license The license object.
		 * @param array $args     The activation arguments (added in 3.9.0).
		 */
		$result = apply_filters( 'edd_sl_activate_license_response', $this->get_activation_result( $license, $args ), $license, $args );

		if ( $result['success'] ) {

			// Activate the site for the license.
			$license->add_site( $args['url'], $args['environment'] );
			if ( $this->is_local_url( $args['url'], $args['environment'] ) ) {
				$result['is_local'] = true;
			}

			if ( apply_filters( 'edd_sl_log_license_activation', true ) ) {
				edd_software_licensing()->log_license_activation( $license->id, $_SERVER );
			}

			do_action( 'edd_sl_activate_license', $license->ID, $license->download_id );
		}

		if ( false !== $license ) {
			// All good, give some additional info about the activation.
			$result['license_limit'] = $license->activation_limit;
			$result['site_count']    = $license->activation_count;
			$result['expires']       = $license->expiration;
			// Just leaving this in here in case others are using it.
			if ( $license->activation_limit > 0 ) {
				$result['activations_left'] = $license->activation_limit - $license->activation_count;
			} else {
				$result['activations_left'] = 'unlimited';
			}
		}

		/**
		 * Allow developers to filter the activation result before it is returned.
		 *
		 * @param array $result   The activation result.
		 * @param array $args     The activation arguments.
		 * @param object $license The license object (added in 3.9.0).
		 */
		return apply_filters( 'edd_sl_post_activate_license_result', $result, $args, $license );
	}

	/**
	 * Gets the license activation data.
	 *
	 * @param false|\EDD\SoftwareLicensing\Licenses\License $license The license object, if available.
	 * @param array                                         $args    The activation arguments.
	 * @return array
	 */
	private function get_activation_result( $license, $args ) {
		// This license does not even exist.
		if ( empty( $license ) ) {
			return array(
				'success' => false,
				'error'   => 'missing',
			);
		}

		// No item ID or name provided.
		if ( empty( $args['item_id'] ) && empty( $args['item_name'] ) ) {
			return array(
				'success' => false,
				'error'   => 'missing_item_id',
			);
		}

		// License key revoked.
		if ( 'disabled' === $license->status ) {
			return array(
				'success' => false,
				'error'   => 'disabled',
			);
		}

		// Keys don't match.
		if ( $args['key'] !== $license->key ) {
			return array(
				'success' => false,
				'error'   => 'key_mismatch',
			);
		}

		// This license has expired.
		if ( ! $license->is_lifetime && $license->expiration < $args['expiration'] ) {
			return array(
				'success' => false,
				'error'   => 'expired',
				'expires' => $license->expiration,
			);
		}

		if ( empty( $args['url'] ) && ! edd_software_licensing()->force_increase( $license->ID ) ) {
			return array(
				'success' => false,
				'error'   => 'missing_url',
			);
		}

		// Trying to activate bundle license.
		if ( ! apply_filters( 'edd_sl_allow_bundle_activation', false, $license ) && $license->get_download()->is_bundled_download() ) {
			return array(
				'success' => false,
				'error'   => 'bundle_activation_not_allowed',
			);
		}

		// No activations left.
		if ( $license->is_at_limit() && ! $this->is_local_url( $args['url'], $args['environment'] ) && ( edd_software_licensing()->force_increase() || ! $license->is_site_active( $args['url'] ) ) ) {
			return array(
				'success'   => false,
				'error'     => 'no_activations_left',
				'max_sites' => $license->activation_count,
			);
		}

		if ( ! empty( $args['item_id'] ) && ! edd_software_licensing()->is_download_id_valid_for_license( $args['item_id'], $args['key'] ) ) {
			return array(
				'success' => false,
				'error'   => 'invalid_item_id',
			);
		}

		if ( ! empty( $args['item_name'] ) ) {
			$item_name = html_entity_decode( $args['item_name'] );
			if ( ( ! defined( 'EDD_BYPASS_NAME_CHECK' ) || ! EDD_BYPASS_NAME_CHECK ) && ! edd_software_licensing()->check_item_name( $license->download_id, $item_name, $license ) ) {
				return array(
					'success' => false,
					'error'   => 'item_name_mismatch',
				);
			}
		}

		return array(
			'success' => true,
		);
	}

	/**
	 * Checks if the URL is a local URL.
	 *
	 * @param string $url The URL to check.
	 * @param string $environment The environment to check.
	 * @return bool
	 */
	private function is_local_url( $url, $environment ): bool {
		$bypass_local = edd_get_option( 'edd_sl_bypass_local_hosts', false );
		if ( empty( $bypass_local ) ) {
			return false;
		}

		return edd_software_licensing()->is_local_url( $url, $environment );
	}
}
