<?php
/**
 * The legacy License Activations DB Class
 *
 * @package   EDD\SoftwareLicensing\Legacy\Database
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Legacy\Database;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\SoftwareLicensing\Database\Queries\LicenseActivation as Query;

/**
 * The License Activations DB Class
 *
 * @since  3.6
 */
class Activations extends Base {

	/**
	 * Get things started
	 *
	 * @access  public
	 * @since   3.6
	 */
	public function __construct() {

		global $wpdb;

		$this->table_name  = $wpdb->prefix . 'edd_license_activations';
		$this->primary_key = 'id';
		$this->version     = '1.1';
	}

	/**
	 * Get columns and formats
	 *
	 * @access  public
	 * @since   3.6
	 */
	public function get_columns() {
		return array(
			'id'           => '%d',
			'site_name'    => '%s',
			'license_id'   => '%d',
			'activated'    => '%d',
			'is_local'     => '%d',
			'date_created' => '%s',
			'last_request' => '%s',
		);
	}

	/**
	 * Returns the column labels only.
	 *
	 * @since 3.6
	 * @return array
	 */
	public function get_column_labels() {
		return array_keys( $this->get_columns() );
	}

	/**
	 * Get default column values
	 *
	 * @access  public
	 * @since   3.6
	 */
	public function get_column_defaults() {
		return array(
			'site_name'    => null,
			'license_id'   => null,
			'activated'    => 1,
			'is_local'     => 0,
			'date_created' => null,
			'last_request' => null,
		);
	}

	/**
	 * Retrieve all activations for a license.
	 *
	 * @access  public
	 * @since   3.6
	 */
	public function get_activations( $args = array() ) {
		$defaults = array(
			'number'    => 25,
			'offset'    => 0,
			'search'    => '',
			'orderby'   => 'license_id',
			'order'     => 'ASC',
			'activated' => 1,
		);

		$args  = wp_parse_args( $args, $defaults );
		$query = new Query();

		return $query->query( $args );
	}

	/**
	 * Delete any license activations.
	 *
	 * @since 3.6
	 * @param int $license_id The ID of the license to delete activations for.
	 *
	 * @return bool
	 */
	public function delete_all_activations( $license_id ) {
		$query       = new Query();
		$activations = $query->query(
			array(
				'license_id' => $license_id,
				'number'     => 99999,
			)
		);

		foreach ( $activations as $activation ) {
			$query->delete_item( $activation->id );
		}

		return true;
	}

	/**
	 * Count the total number of licenses in the database
	 *
	 * @access  public
	 * @since   3.6
	 * @param array $args The arguments for the query.
	 * @return int The total number of activations.
	 */
	public function count( $args = array() ) {
		$args['count'] = true;
		$query         = new Query( $args );

		return absint( $query->found_items );
	}

	/**
	 * Delete a license activation.
	 *
	 * @since 3.9.0
	 * @param int $site_id The ID of the site to delete.
	 * @return bool
	 */
	public function delete( $site_id = '' ) {
		$query = new Query();

		return $query->delete_item( $site_id );
	}

	/**
	 * Insert a new license activation.
	 * This overrides the parent method to use the new query class.
	 *
	 * @since 3.9.0
	 * @param array  $data The data to insert.
	 * @param string $type The type of data to insert (not used).
	 */
	public function insert( $data = array(), $type = '' ) {
		$query = new Query();

		return $query->add_item( $data );
	}

	/**
	 * Update a license activation.
	 * This overrides the parent method to use the new query class.
	 *
	 * @since 3.9.0
	 * @param int    $id    The ID of the item to update.
	 * @param array  $data  The data to update.
	 * @param string $where The where clause (not used).
	 */
	public function update( $id, $data = array(), $where = '' ) {
		$query = new Query();

		return $query->update_item( $id, $data );
	}

	/**
	 * Create the table
	 *
	 * @since   3.6
	 * @deprecated 3.9.0
	 */
	public function create_table() {}
}
