<?php
/**
 * LowKeys email template.
 *
 * @package EDD/SoftwareLicensing
 * @subpackage Emails
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Emails\Templates;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Emails\Templates\EmailTemplate;

/**
 * Class LowKeys
 *
 * @since 3.9.0
 * @package EDD\SoftwareLicensing\Templates\Emails
 */
class LowKeys extends EmailTemplate {

	/**
	 * Whether this email can be previewed.
	 *
	 * @since 3.9.0
	 * @var bool
	 */
	protected $can_preview = true;

	/**
	 * Whether this email can be tested.
	 *
	 * @since 3.9.0
	 * @var bool
	 */
	protected $can_test = true;

	/**
	 * Unique identifier for this template.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $email_id = 'sl_low_keys';

	/**
	 * The email recipient.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $recipient = 'admin';

	/**
	 * The email context.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $context = 'download';

	/**
	 * Retrieves the email sender.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $sender = 'software_licensing';

	/**
	 * The email meta.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $meta = array(
		'low_key_threshold' => null,
	);

	/**
	 * Name of the template.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_name(): string {
		return __( 'Preset License Keys Warning', 'edd_sl' );
	}

	/**
	 * Description of the email.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_description(): string {
		return sprintf(
			/* translators: %d: Low key threshold */
			__( 'This email is sent when the number of remaining preset license keys is %d.', 'edd_sl' ),
			edd_get_email_meta( $this->get_email()->id, 'low_key_threshold', true ) ?: 10 // phpcs:ignore Universal.Operators.DisallowShortTernary.Found
		);
	}

	/**
	 * Gets the email defaults.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function defaults(): array {
		return array(
			'subject' => __( 'License Key Stock Warning', 'edd_sl' ),
			'content' => __( 'The number of remaining preset license keys for {product_name} is low.', 'edd_sl' ),
			'status'  => 0,
		);
	}

	/**
	 * Gets the editable properties for this email.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	protected function get_editable_properties(): array {
		return array(
			'subject',
			'content',
			'status',
		);
	}

	/**
	 * Gets the preview data.
	 *
	 * @since 3.9.0
	 * @return array|bool
	 */
	protected function get_preview_data() {
		$downloads = new \WP_Query(
			array(
				'post_type'      => 'download',
				'posts_per_page' => 1,
				'orderby'        => 'rand',
				'meta_query'     => array(
					array(
						'key'     => '_edd_sl_keys',
						'compare' => 'EXISTS',
					),
				),
			)
		);
		if ( empty( $downloads->found_posts ) ) {
			return false;
		}

		$download = reset( $downloads->posts );

		return array(
			$download->ID,
			new \EDD_SL_Download( $download->ID ),
		);
	}
}
