<?php
/**
 * License Activations Table.
 *
 * @package     EDD\SoftwareLicensing\Database\Tables
 * @copyright   Copyright (c) 2024, Easy Digital Downloads, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Database\Tables;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Database\Table;

/**
 * Setup the global "edd_activations" database table
 *
 * @since 3.9.0
 */
final class LicenseActivations extends Table {

	/**
	 * Table name.
	 *
	 * @access protected
	 * @since 3.9.0
	 * @var string
	 */
	protected $name = 'license_activations';

	/**
	 * Database version.
	 *
	 * @access protected
	 * @since 3.9.0
	 * @var int
	 */
	protected $version = 202508070;

	/**
	 * Array of upgrade versions and methods.
	 *
	 * @since 3.9.0
	 * @var array
	 */
	protected $upgrades = array(
		'202508070' => 202508070,
	);

	/**
	 * Setup the database schema.
	 *
	 * @access protected
	 * @since 3.9.0
	 * @return void
	 */
	protected function set_schema() {
		$this->schema = '
			id bigint(20) NOT NULL AUTO_INCREMENT,
			site_name varchar(255) NOT NULL,
			license_id bigint(20) NOT NULL,
			activated TINYINT(1) NOT NULL,
			is_local TINYINT(1) NOT NULL,
			date_created datetime,
			last_request datetime,
			PRIMARY KEY (id),
			UNIQUE KEY site_name (site_name,license_id),
			KEY license_id_activated (license_id,activated)
		';
	}

	/**
	 * Upgrade routine for version 202409241.
	 *
	 * @since 3.9.0
	 * @return bool
	 */
	public function __202508070() {

		$checks = array();
		// Delete old/irrelevant database options.
		delete_option( $this->table_prefix . 'edd_license_activations_db_version' );
		delete_option( 'wp_edd_license_activations_db_version' );

		$checks['site_name'] = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY COLUMN site_name VARCHAR(255) NOT NULL" );

		// Rename the site_id column to id for consistency with WordPress Core.
		if ( $this->column_exists( 'site_id' ) ) {
			$checks['site_id'] = $this->get_db()->query( "ALTER TABLE {$this->table_name} CHANGE COLUMN site_id id bigint(20) NOT NULL AUTO_INCREMENT" );
		}

		// add the date_created column.
		if ( ! $this->column_exists( 'date_created' ) ) {
			$checks['date_created'] = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD COLUMN `date_created` datetime DEFAULT NULL AFTER `is_local`;" );
		}

		// add the last_request column.
		if ( ! $this->column_exists( 'last_request' ) ) {
			$checks['last_request'] = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD COLUMN `last_request` datetime DEFAULT NULL AFTER `date_created`;" );
		}

		if ( in_array( false, $checks, true ) ) {
			return $this->is_success( false );
		}

		return $this->is_success( true );
	}
}
