<?php
/**
 * Activation Query Class.
 *
 * @package   EDD\SoftwareLicensing\Database\Queries
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Database\Queries;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Database\Query;

/**
 * Class used for querying items.
 *
 * @since 3.9.0
 *
 * @see \EDD\Database\Queries\Query::__construct() for accepted arguments.
 */
class LicenseActivation extends Query {

	/** Table Properties ******************************************************/

	/**
	 * Name of the database table to query.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $table_name = 'license_activations';

	/**
	 * String used to alias the database table in MySQL statement.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $table_alias = 'la';

	/**
	 * Name of class used to setup the database schema
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $table_schema = '\\EDD\\SoftwareLicensing\\Database\\Schemas\\LicenseActivations';

	/** Item ******************************************************************/

	/**
	 * Name for a single item
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $item_name = 'license_activation';

	/**
	 * Plural version for a group of items.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $item_name_plural = 'license_activations';

	/**
	 * Callback function for turning IDs into objects
	 *
	 * @since 3.9.0
	 * @access public
	 * @var mixed
	 */
	protected $item_shape = '\\EDD\\SoftwareLicensing\\Licenses\\Activation';

	/** Cache *****************************************************************/

	/**
	 * Group to cache queries and queried items in.
	 *
	 * @since 3.9.0
	 * @access public
	 * @var string
	 */
	protected $cache_group = 'license-activation';

	/**
	 * Add an activation to the database.
	 *
	 * @since 3.9.0
	 * @param array $data The data to add.
	 * @return int|bool
	 */
	public function add_item( $data = array() ) {
		if ( empty( $data['site_name'] ) ) {
			return false;
		}

		$data['site_name'] = untrailingslashit( edd_software_licensing()->clean_site_url( $data['site_name'] ) );

		return parent::add_item( $data );
	}

	/**
	 * Update an activation in the database.
	 *
	 * @since 3.9.0
	 * @param int   $item_id The activation ID.
	 * @param array $data The data to update.
	 * @return bool
	 */
	public function update_item( $item_id = 0, $data = array() ) {
		if ( empty( $item_id ) ) {
			return false;
		}
		if ( ! empty( $data['site_name'] ) ) {
			$data['site_name'] = untrailingslashit( edd_software_licensing()->clean_site_url( $data['site_name'] ) );
		}

		return parent::update_item( $item_id, $data );
	}

	/**
	 * Deletes an activation from the database.
	 * This also deletes the activation data from the license activations data table.
	 *
	 * @since 3.9.0
	 * @param int $item_id The activation ID.
	 * @return bool
	 */
	public function delete_item( $item_id = 0 ) {
		if ( empty( $item_id ) ) {
			return false;
		}

		$deleted = parent::delete_item( $item_id );
		if ( ! $deleted ) {
			return false;
		}

		$activations_data = new LicenseActivationsData();
		$data             = $activations_data->query(
			array(
				'activation_id' => $item_id,
			)
		);
		if ( ! empty( $data ) ) {
			foreach ( $data as $activation ) {
				$activations_data->delete_item( $activation->id );
			}
		}

		return $deleted;
	}

	/**
	 * Query the database for activations.
	 *
	 * @since 3.9.0
	 * @param array $args The arguments to pass to the query.
	 * @return array
	 */
	public function query( $args = array() ) {

		// Account for 'paged' in legacy $args.
		if ( isset( $args['paged'] ) && $args['paged'] > 1 ) {
			$number         = isset( $args['number'] ) ? $args['number'] : 25;
			$args['offset'] = ( ( $args['paged'] - 1 ) * $number );
			unset( $args['paged'] );
		}

		// Parse the legacy status argument.
		if ( isset( $args['status'] ) ) {
			$status = $this->parse_status( $args['status'] );
			if ( ! empty( $status ) ) {
				$args = array_merge( $args, $status );
			}
			unset( $args['status'] );
		}

		if ( isset( $args['number'] ) && $args['number'] < 1 ) {
			$args['number'] = 999999999999;
		}

		if ( ! empty( $args['site_name'] ) ) {
			if ( is_string( $args['site_name'] ) ) {
				$args['site_name__in'] = array( trailingslashit( $args['site_name'] ), untrailingslashit( $args['site_name'] ) );
			} else {
				$args['site_name__in'] = $args['site_name'];
			}
			unset( $args['site_name'] );
		}

		if ( ! empty( $args['site_id'] ) ) {
			if ( is_array( $args['site_id'] ) ) {
				$args['id__in'] = $args['site_id'];
			} else {
				$args['id'] = $args['site_id'];
			}
			unset( $args['site_id'] );
		}

		// Specific Licenses(s).
		if ( isset( $args['license_id'] ) && is_array( $args['license_id'] ) ) {
			$args['license_id__in'] = $args['license_id'];
			unset( $args['license_id'] );
		}

		// Specific activation status.
		if ( isset( $args['activated'] ) ) {
			if ( is_array( $args['activated'] ) ) {
				$args['activated__in'] = $args['activated'];
			}
			if ( isset( $args['activated__in'] ) ) {
				unset( $args['activated'] );
			}
		}

		// Specific is_local Checks.
		if ( isset( $args['is_local'] ) ) {
			if ( is_array( $args['is_local'] ) ) {
				$args['is_local__in'] = $args['is_local'];
				unset( $args['is_local'] );
			}
		}

		// Set up filters for JOIN clauses if needed.
		$query_clauses_filters = $this->get_query_clauses_filters( $args );
		foreach ( $query_clauses_filters as $filter ) {
			if ( $filter['condition'] ) {
				add_filter( 'edd_license_activations_query_clauses', array( $this, $filter['callback'] ) );
			}
		}

		$result = parent::query( $args );

		// Remove filters after query.
		foreach ( $query_clauses_filters as $filter ) {
			if ( $filter['condition'] ) {
				remove_filter( 'edd_license_activations_query_clauses', array( $this, $filter['callback'] ) );
			}
		}

		return $result;
	}

	/**
	 * Get the array of possible query clause filters.
	 *
	 * @since 3.9.1
	 * @param array $query The query arguments.
	 * @return array
	 */
	private function get_query_clauses_filters( $query ) {
		return array(
			array(
				'condition' => ! empty( $query['download_id'] ) || ! empty( $query['customer_id'] ) || ! empty( $query['customer_id__in'] ),
				'callback'  => 'query_by_license_table',
			),
		);
	}

	/**
	 * Filter the query clause to join the licenses table and filter by download_id and/or customer_id.
	 *
	 * @since 3.9.1
	 * @access public
	 *
	 * @param array $clauses The clauses which will generate the final SQL query.
	 * @return array
	 */
	public function query_by_license_table( $clauses ) {
		global $wpdb;

		$primary_alias  = $this->table_alias;
		$primary_column = parent::get_primary_column_name();

		$licenses_query = new License();
		$join_alias     = $licenses_query->table_alias;
		$license_pk     = $licenses_query->get_primary_column_name();

		// Build the JOIN conditions.
		$join_conditions = array();
		$prepare_args    = array();

		// Base join condition.
		$base_condition = "{$primary_alias}.license_id = {$join_alias}.{$license_pk}";

		// Add download_id condition if present.
		if ( ! empty( $this->query_vars['download_id'] ) ) {
			$join_conditions[] = "{$join_alias}.download_id = %d";
			$prepare_args[]    = absint( $this->query_vars['download_id'] );
		}

		// Add customer_id condition if present.
		if ( ! empty( $this->query_vars['customer_id'] ) ) {
			$join_conditions[] = "{$join_alias}.customer_id = %d";
			$prepare_args[]    = absint( $this->query_vars['customer_id'] );
		}

		// Add customer_id__in condition if present.
		if ( ! empty( $this->query_vars['customer_id__in'] ) && is_array( $this->query_vars['customer_id__in'] ) ) {
			$customer_ids      = array_map( 'absint', $this->query_vars['customer_id__in'] );
			$placeholders      = implode( ', ', array_fill( 0, count( $customer_ids ), '%d' ) );
			$join_conditions[] = "{$join_alias}.customer_id IN ({$placeholders})";
			$prepare_args      = array_merge( $prepare_args, $customer_ids );
		}

		// Build the full JOIN clause.
		$conditions_sql = ! empty( $join_conditions ) ? ' AND ' . implode( ' AND ', $join_conditions ) : '';
		$join_sql       = " INNER JOIN {$licenses_query->table_name} {$join_alias} ON ({$base_condition}{$conditions_sql})";

		// Prepare the query if we have parameters.
		if ( ! empty( $prepare_args ) ) {
			// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
			$clauses['join'] .= $wpdb->prepare( $join_sql, $prepare_args );
			// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
		} else {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$clauses['join'] .= $join_sql;
		}

		return $clauses;
	}

	/**
	 * Parse the legacy status argument.
	 *
	 * @since 3.9.1
	 * @param string $status The status to parse.
	 * @return array
	 */
	private function parse_status( $status ) {

		if ( in_array( $status, array( 'activated', 1, '1' ), true ) ) {
			return array( 'activated' => 1 );
		}

		if ( in_array( $status, array( 'deactivated', 0, '0' ), true ) ) {
			return array( 'activated' => 0 );
		}

		if ( 'all' === $status ) {
			return array( 'activated__in' => array( 0, 1 ) );
		}

		return array();
	}
}
