<?php
/**
 * License expiration related cron events.
 *
 * @package     EDD\SoftwareLicensing\Cron\Components
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Cron\Components;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Cron\Events\SingleEvent;
use EDD\Cron\Components\Component;

/**
 * Expire Class
 *
 * @since 3.9.0
 */
class Expire extends Component {

	/**
	 * The unique identifier for this component.
	 *
	 * @var string
	 */
	protected static $id = 'sl_expirations';

	/**
	 * The number of licenses to process per step.
	 *
	 * @var int
	 */
	private $per_step = 100;

	/**
	 * Gets the array of subscribed events.
	 */
	public static function get_subscribed_events(): array {
		return array(
			'edd_sl_daily_scheduled_events' => 'check_licenses',
			'edd_sl_expire_licenses'        => array( 'check_licenses', 10, 1 ),
		);
	}

	/**
	 * Check for expired licenses.
	 *
	 * @since 3.9.0
	 * @param int $step The step to get the next set of licenses.
	 */
	public function check_licenses( int $step = 0 ): void {
		if ( ! edd_doing_cron() ) {
			return;
		}

		$licenses = edd_software_licensing()->licenses_db->get_licenses( $this->get_args( $step ) );
		if ( ! $licenses ) {
			return;
		}

		foreach ( $licenses as $license ) {
			$license->status = 'expired';
		}

		++$step;
		SingleEvent::add( time() + ( 3 * MINUTE_IN_SECONDS ), 'edd_sl_expire_licenses', array( $step ) );
	}

	/**
	 * Get the arguments for the query.
	 *
	 * @since 3.9.0
	 * @param int $step The step to get the next set of licenses.
	 * @return array
	 */
	private function get_args( int $step ): array {
		$args = array(
			'number'     => $this->per_step,
			'parent'     => 0, // Child keys get expired during set_license_status().
			'expiration' => array(
				'start' => strtotime( '-1 month' ),
				'end'   => current_time( 'timestamp' ),
			),
			'status'     => array( 'active', 'inactive' ),
			'offset'     => $step * $this->per_step,
		);

		return apply_filters( 'edd_sl_expired_licenses_args', $args );
	}
}
