<?php

namespace EDD\SoftwareLicensing\Checkout;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Cart class.
 *
 * @since 3.9.0
 */
class Cart implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events(): array {
		return array(
			'edd_checkout_cart_item_title_after' => 'show_license_expiration',
		);
	}

	/**
	 * Adds the product expiration date to the cart item title.
	 *
	 * @param array $item The cart item.
	 * @since 3.9.0
	 */
	public function show_license_expiration( $item ) {
		if ( ! edd_get_option( 'edd_sl_show_cart_item_expiration', false ) ) {
			return;
		}

		$price_id   = isset( $item['options']['price_id'] ) && is_numeric( $item['options']['price_id'] ) ? (int) $item['options']['price_id'] : false;
		$expiration = edd_sl_get_product_expiration_date( $item['id'], $price_id, $this->get_upgrade_args( $item ) );
		if ( ! $expiration ) {
			return;
		}

		$expiration = 'lifetime' === $expiration ? __( 'Lifetime', 'edd_sl' ) : edd_date_i18n( $expiration, get_option( 'date_format' ) );
		?>
		<div class="edd-sl-details edd-sl-details-cart">
			<span class="edd-sl-label"><?php esc_html_e( 'Expiration', 'edd_sl' ); ?>:</span>
			<span class="edd-sl-key"><?php echo esc_html( $expiration ); ?></span>
		</div>
		<?php
	}

	/**
	 * Get the upgrade args.
	 *
	 * @param array $item The cart item.
	 * @since 3.9.0
	 * @return array
	 */
	private function get_upgrade_args( $item ): array {
		if ( empty( $item['options']['license_id'] ) ) {
			return array();
		}

		return array(
			'license_id'  => $item['options']['license_id'],
			'download_id' => $item['id'],
			'price_id'    => isset( $item['options']['price_id'] ) ? (int) $item['options']['price_id'] : null,
		);
	}
}
