<?php
/**
 * Software Licensing Reports Products Trait
 *
 * @package   EDD\SoftwareLicensing\Admin\Reports\Traits
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Reports\Traits;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Reports;

/**
 * Products Trait
 *
 * @since 3.9.0
 */
trait Products {

	/**
	 * Maybe adds the filter for the products dropdown.
	 * We have to check the report ID because the action is run for a lot of different contexts.
	 *
	 * @since 3.9.0
	 * @param mixed $report
	 * @return void
	 */
	public function maybe_filter_products( $report ) {
		if ( empty( $report->object_id ) || $this->id !== $report->object_id ) {
			return;
		}
		add_filter( 'edd_report_filters', array( $this, 'replace_product_filter' ) );
	}

	/**
	 * Removes the filter for the products dropdown.
	 * We have to check the report ID because the action is run for a lot of different contexts.
	 *
	 * @since 3.9.0
	 * @param mixed $report
	 * @return void
	 */
	public function remove_filter_products( $report ) {
		if ( empty( $report->object_id ) || $this->id !== $report->object_id ) {
			return;
		}
		remove_filter( 'edd_report_filters', array( $this, 'replace_product_filter' ) );
	}

	/**
	 * Replaces the product filter with a custom one.
	 *
	 * @since 3.9.0
	 * @param array $filters
	 * @return array
	 */
	public function replace_product_filter( $filters ) {
		$filters['products']['display_callback'] = array( $this, 'product_filter' );

		return $filters;
	}

	/**
	 * Outputs the product filter.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	public function product_filter() {
		$products = Reports\get_filter_value( 'products' );
		add_filter( 'edd_product_dropdown_args', array( $this, 'filter_product_dropdown' ) );

		$select = new \EDD\HTML\ProductSelect(
			array(
				'chosen'           => true,
				'variations'       => false,
				'selected'         => empty( $products ) ? 0 : $products,
				'show_option_none' => false,
				/* translators: %s: label plural */
				'show_option_all'  => sprintf( __( 'All %s', 'edd_sl' ), edd_get_label_plural() ),
			)
		); ?>

		<span class="edd-graph-filter-options graph-option-section">
			<?php $select->output(); ?>
		</span>
		<?php
		remove_filter( 'edd_product_dropdown_args', array( $this, 'filter_product_dropdown' ) );
	}

	/**
	 * Filters the product dropdown to only show products with software licensing enabled.
	 *
	 * @since 3.9.0
	 * @param array $args
	 * @return array
	 */
	public function filter_product_dropdown( $args ) {
		$args['meta_query'] = array(
			array(
				'key'     => '_edd_sl_enabled',
				'value'   => '1',
				'compare' => '=',
			),
		);

		return $args;
	}
}
