<?php
/**
 * Product Versions Pie Chart
 *
 * @package     EDD\SoftwareLicensing\Admin\Reports\Endpoints\Pies
 * @copyright   Copyright (c) 2025, Easy Digital Downloads, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Reports\Endpoints\Pies;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Reports\Endpoints\Pies\Pie;

/**
 * Product Versions Pie Chart class.
 *
 * Builds pie chart data for product versions using the Pie abstract pattern.
 *
 * @since 3.9.0
 */
class Versions extends Pie {
	use \EDD\SoftwareLicensing\Admin\Reports\Traits\Query;

	/**
	 * The key for the dataset.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $key = 'versions';

	/**
	 * The date column for the query.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $date_column = 'date_modified';

	/**
	 * The versions for the query.
	 *
	 * @since 3.9.0
	 * @var array
	 */
	private $versions;

	/**
	 * Gets the chart endpoint ID.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	protected function get_id(): string {
		return 'sl_product_versions';
	}

	/**
	 * Gets the chart label for display.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	protected function get_label(): string {
		/* translators: %s: product name */
		return sprintf( __( '%s Versions', 'edd_sl' ), edd_get_download_name( $this->get_product() ) );
	}

	/**
	 * Registers this chart with the reports system.
	 *
	 * @since 3.9.0
	 */
	protected function register(): void {
		if ( empty( $this->get_data() ) ) {
			return;
		}

		parent::register();
	}

	/**
	 * Gets the query results for building the chart.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	protected function get_query_results(): array {
		if ( ! is_null( $this->versions ) ) {
			return $this->versions;
		}

		global $wpdb;

		$this->versions = $wpdb->get_results(
			"SELECT product_version as piece, COUNT(*) as value FROM {$wpdb->prefix}edd_license_activations_data
			 WHERE {$this->get_date_sql()} {$this->get_product_sql_query()}
			 GROUP BY piece"
		);

		return $this->versions;
	}
}
