<?php
/**
 * Product Versions Pie Chart
 *
 * @package     EDD\SoftwareLicensing\Admin\Reports\Endpoints\Pies
 * @copyright   Copyright (c) 2025, Easy Digital Downloads, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Reports\Endpoints\Pies;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\Reports\Endpoints\Pies\Pie;

/**
 * Meta Key Pie Chart class.
 *
 * Builds pie chart data for meta keys using the Pie abstract pattern.
 *
 * @since 3.9.0
 */
class Meta extends Pie {
	use \EDD\SoftwareLicensing\Admin\Reports\Traits\Query;

	/**
	 * The key for the dataset.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $key = 'versions';

	/**
	 * The date column for the query.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $date_column = 'date_modified';

	/**
	 * The pieces for the query.
	 *
	 * @since 3.9.0
	 * @var array
	 */
	private $pieces;

	/**
	 * The versions for the query.
	 *
	 * @since 3.9.0
	 * @var array
	 */
	private $meta_key;

	/**
	 * Constructor.
	 *
	 * @since 3.9.0
	 */
	public function __construct( $reports, $meta_key ) {
		$this->meta_key = $meta_key;
		parent::__construct( $reports );
	}

	/**
	 * Gets the chart endpoint ID.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	protected function get_id(): string {
		return 'sl_meta_key_' . $this->meta_key;
	}

	/**
	 * Gets the chart label for display.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	protected function get_label(): string {
		$meta_keys = eddsl_get_activation_data_meta_keys();
		if ( ! array_key_exists( $this->meta_key, $meta_keys ) ) {
			return '';
		}

		return $meta_keys[ $this->meta_key ];
	}

	/**
	 * Registers this chart with the reports system.
	 *
	 * @since 3.9.0
	 */
	protected function register(): void {
		if ( empty( $this->get_data() ) ) {
			return;
		}

		parent::register();
	}

	/**
	 * Gets the query results for building the chart.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	protected function get_query_results(): array {
		if ( ! is_null( $this->pieces ) ) {
			return $this->pieces;
		}

		global $wpdb;

		$activation_data = "SELECT activation_id FROM {$wpdb->prefix}edd_license_activations_data
			 WHERE {$this->get_date_sql()} {$this->get_product_sql_query()}";

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$sql = $wpdb->prepare(
			"SELECT SUBSTRING_INDEX(SUBSTRING_INDEX(meta_value, '.', 2), '-', 1) as piece, COUNT(*) as value FROM {$wpdb->prefix}edd_license_activationmeta
			 WHERE meta_key = %s
			 AND edd_license_activation_id IN ( $activation_data )
			 GROUP BY SUBSTRING_INDEX(SUBSTRING_INDEX(meta_value, '.', 2), '-', 1)",
			$this->meta_key
		);
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$this->pieces = $wpdb->get_results( $sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

		return $this->pieces;
	}
}
