<?php
/**
 * Retroactive Licensing tool.
 *
 * @package   EDD\SoftwareLicensing\Admin\Licensing
 * @copyright Copyright (c) Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Licensing;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * This class handles generating license keys for purchases made before Software Licensing was activated
 */
class Retroactive implements SubscriberInterface {
	use \EDD\SoftwareLicensing\Deprecated\Retroactive;

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_tools_recount_stats_before' => 'tool_box',
			'edd_download_batch_export'      => array( 'modify_request', -100 ),
		);
	}

	/**
	 * Render the admin UI under Downloads > Tools
	 *
	 * @access public
	 * @since 2.4
	 * @return void
	 */
	public function tool_box() {
		if ( ! current_user_can( 'manage_licenses' ) ) {
			return;
		}
		?>
		<div class="postbox edd-retroactive-licensing">
			<h2 class="hndle"><?php esc_html_e( 'Software Licensing - Fix Missing Licenses', 'edd_sl' ); ?></h3>
			<div class="inside">
				<p><?php esc_html_e( 'Use this tool to provision missing licenses for orders. It\'s perfect for scenarios such as adding licensing support to a product that previously didn\'t have it, adding a product to a licensed bundle, or when licenses were not generated for an order.', 'edd_sl' ); ?></p>
				<p><?php esc_html_e( 'A CSV file of the processed orders can be downloaded when processing is complete.', 'edd_sl' ); ?></p>
				<form id="edd-retroactive-processor" class="edd-export-form edd-import-export-form">
					<?php
					$this->export_form( 'edd_retroactive_licenses' );
					wp_nonce_field( 'edd_ajax_export', 'edd_ajax_export' );
					?>
					<input type="hidden" name="edd-export-class" value="EDD\SoftwareLicensing\Admin\Exports\Retroactive">
					<button type="submit" class="button button-secondary"><?php esc_html_e( 'Process Licenses', 'edd_sl' ); ?></button>
				</form>
			</div>
		</div>
		<?php
	}

	/**
	 * Render the export form for the retroactive licensing processor.
	 *
	 * @since 3.9.0
	 * @param string $exporter_id The exporter ID.
	 * @return void
	 */
	public function export_form( $exporter_id ) {
		if ( 'edd_retroactive_licenses' !== $exporter_id ) {
			return;
		}
		wp_enqueue_script( 'edd-admin-tools-export' );
		?>
		<div class="sl-retro-download-selector">
			<select name="sl_retro_type" id="sl-retro-type">
				<option value="all">
					<?php
					/* translators: %s: plural download name */
					printf( esc_html__( 'All %s', 'edd_sl' ), esc_html( edd_get_label_plural() ) );
					?>
				</option>
				<option value="single">
					<?php
					/* translators: %s: singular download name */
					printf( esc_html__( 'Single %s', 'edd_sl' ), esc_html( edd_get_label_singular() ) );
					?>
				</option>
			</select>
			<div id="sl-retro-single-wrapper" style="display:none;">
				<?php
				add_filter( 'edd_product_dropdown_args', array( $this, 'filter_product_dropdown' ) );
				$select = new \EDD\HTML\ProductSelect(
					array(
						'chosen'  => true,
						'bundles' => false,
						'name'    => 'edd_sl_single_id',
						'id'      => 'edd-sl-single-id',
					)
				);
				$select->output();
				remove_filter( 'edd_product_dropdown_args', array( $this, 'filter_product_dropdown' ) );
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Filter the product dropdown to only show products with Software Licensing enabled.
	 *
	 * @since 3.9.0
	 * @param array $product_args The product arguments.
	 * @return array
	 */
	public function filter_product_dropdown( $product_args ) {
		$product_args['meta_key']   = '_edd_sl_enabled';
		$product_args['meta_value'] = '1';

		return $product_args;
	}

	/**
	 * This is really hacky, but important until EDD core better supports namespaced classes.
	 * During the download, `EDD\Admin\Export` get converted to:
	 * `EDD\\\\Invoices\\\\Admin\\\\Export` . This breaks the `class_exists()`
	 * check in EDD core ( @see edd_process_batch_export_download() ). Therefore, we hook in before
	 * that runs to strip slashes, which makes everything work again. Dumb, but temporarily necessary.
	 *
	 * @link https://github.com/awesomemotive/easy-digital-downloads-pro/issues/1150
	 * @since 3.9.0
	 */
	public function modify_request() {
		if ( isset( $_REQUEST['class'] ) && \EDD\SoftwareLicensing\Admin\Exports\Retroactive::class === stripslashes( $_REQUEST['class'] ) ) {
			$_REQUEST['class'] = \EDD\SoftwareLicensing\Admin\Exports\Retroactive::class;
		}
	}
}
