<?php
/**
 * Retroactive Licensing Export.
 *
 * @package   EDD\SoftwareLicensing\Admin\Exports
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Exports;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Retroactive Licensing Export.
 */
class Retroactive extends \EDD_Batch_Export {

	use \EDD\SoftwareLicensing\Licenses\Traits\Generator;

	/**
	 * The file type, typically .csv
	 *
	 * @since 3.9.0
	 * @var string
	 */
	public $filetype = '.csv';

	/**
	 * @var string Export type - used for export-specific filters/actions.
	 */
	public $export_type = 'retroactive';

	/**
	 * Used to determine if the export is truly an export or just a batch process.
	 *
	 * @var bool
	 */
	public $is_void = true;

	/**
	 * The number of orders to process per step.
	 *
	 * @since 3.9.0
	 * @var int
	 */
	private $per_step = 30;

	/**
	 * The order IDs to process.
	 *
	 * @var array
	 */
	private $orders = array();

	/**
	 * The product to process.
	 *
	 * @var array
	 */
	private $product;

	/**
	 * The total number of orders to process.
	 *
	 * @var int
	 */
	private $total;

	/**
	 * Process the current step.
	 *
	 * @since 3.9.0
	 * @return bool
	 */
	public function process_step() {
		$processed = parent::process_step();
		if ( $processed ) {
			$this->done = false;

			return true;
		}

		$this->done = true;
		if ( ! empty( $this->orders ) ) {
			$this->message = esc_html__( 'Order processed.', 'edd_sl' );

			return false;
		}

		$generated_keys = $this->get_generated_keys( array() );
		if ( ! empty( $generated_keys ) ) {
			$this->message = sprintf(
				/* translators: %1$d: Number of license keys generated */
				_n(
					'%1$d license was processed. <a href="%2$s" download="%3$s">%4$s</a>',
					'%1$d licenses were processed. <a href="%2$s" download="%3$s">%4$s</a>',
					count( $generated_keys ),
					'edd_sl'
				),
				count( $generated_keys ),
				esc_url( trailingslashit( edd_get_upload_url() ) . 'exports/' . $this->filename ),
				$this->filename,
				esc_html__( 'Download the CSV file.', 'edd_sl' )
			);
		} else {
			$this->message = esc_html__( 'No licenses needed to be processed.', 'edd_sl' );
		}

		return false;
	}

	/**
	 * Get the data to export.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public function get_data() {
		$args           = $this->get_order_args();
		$args['number'] = $this->per_step;
		$args['offset'] = $this->per_step * ( $this->step - 1 );

		add_filter( 'edd_orders_query_clauses', array( $this, 'filter_by_products' ) );
		$orders = edd_get_orders( $args );
		remove_filter( 'edd_orders_query_clauses', array( $this, 'filter_by_products' ) );

		if ( empty( $orders ) ) {
			return false;
		}

		$data = array();
		foreach ( $orders as $order ) {
			$generated_keys = $this->generate_license_keys( $order, $this->product );
			if ( empty( $generated_keys ) ) {
				continue;
			}
			foreach ( (array) $generated_keys as $license_id ) {
				$license = edd_software_licensing()->get_license( $license_id );
				$data[]  = array(
					'id'       => $license->id,
					'key'      => $license->license_key,
					'order_id' => $order->id,
				);
			}
		}

		return $data;
	}

	/**
	 * Override the headers method to prevent the file from being deleted.
	 *
	 * @since 3.9.0
	 */
	public function export() {
		$this->headers();
		edd_die();
	}

	/**
	 * Set the CSV columns.
	 *
	 * @since 2.5
	 *
	 * @return array $cols All the columns.
	 */
	public function csv_cols() {
		return array(
			'id'       => __( 'License ID', 'edd_sl' ),
			'key'      => __( 'License Key', 'edd_sl' ),
			'order_id' => __( 'Order ID', 'edd_sl' ),
		);
	}

	/**
	 * Add a check for the product ID to the query.
	 *
	 * @since 3.9.0
	 * @param array $clauses The query clauses.
	 * @return array
	 */
	public function filter_by_products( $clauses ) {

		if ( ! empty( $this->orders ) ) {
			return $clauses;
		}

		if ( ! empty( $this->product ) ) {
			$products = array( $this->product );
		} else {
			$products = $this->get_licensed_products();
		}
		if ( empty( $products ) ) {
			return $clauses;
		}

		global $wpdb;
		$product_ids       = array_map( 'absint', $products );
		$placeholders      = implode( ',', array_fill( 0, count( $product_ids ), '%d' ) );
		$order_items_query = new \EDD\Database\Queries\Order_Item();

		$condition = $wpdb->prepare(
			"AND {$order_items_query->table_alias}.product_id IN ( {$placeholders} )",
			$product_ids
		);

		$orders_query     = new \EDD\Database\Queries\Order();
		$clauses['join'] .= "
			INNER JOIN {$order_items_query->table_name} {$order_items_query->table_alias} ON(
				{$orders_query->table_alias}.id = {$order_items_query->table_alias}.order_id
				{$condition}
			)";

		return $clauses;
	}

	/**
	 * Return the calculated completion percentage
	 *
	 * @since 3.9.0
	 * @return int
	 */
	public function get_percentage_complete() {
		if ( is_null( $this->total ) ) {
			add_filter( 'edd_orders_query_clauses', array( $this, 'filter_by_products' ) );
			$this->total = edd_count_orders( $this->get_order_args() );
			remove_filter( 'edd_orders_query_clauses', array( $this, 'filter_by_products' ) );
		}
		$percentage = 100;

		if ( $this->total > 0 ) {
			$percentage = ( ( $this->per_step * $this->step ) / $this->total ) * 100;
		}

		if ( $percentage > 100 ) {
			$percentage = 100;
		}

		return $percentage;
	}

	/**
	 * Set the properties specific to the export.
	 *
	 * @since 3.9.0
	 * @param array $request The Form Data passed into the batch processing.
	 */
	public function set_properties( $request ) {
		if ( isset( $request['order_id'] ) ) {
			$this->orders = (array) absint( $request['order_id'] );
		}

		if ( isset( $request['edd_sl_single_id'] ) && 'single' === $request['sl_retro_type'] ) {
			$this->product = absint( $request['edd_sl_single_id'] );
		}

		if ( 1 === $this->step ) {
			$this->delete_transient();
		}
	}

	/**
	 * Whether the current user can export the data.
	 *
	 * @since 3.9.0
	 * @return bool
	 */
	public function can_export() {
		return current_user_can( 'manage_licenses' );
	}

	/**
	 * Gets the parameters for getting orders from the database.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function get_order_args() {
		$args = array(
			'type'       => 'sale',
			'parent'     => 0,
			'status__in' => array( 'complete', 'partially_refunded' ),
		);
		if ( ! empty( $this->orders ) ) {
			$args['id__in'] = $this->orders;
		}

		return $args;
	}

	/**
	 * Get the products that have licensing enabled.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	private function get_licensed_products() {
		$args = array(
			'post_type'     => 'download',
			'nopaging'      => true,
			'fields'        => 'ids',
			'meta_key'      => '_edd_sl_enabled',
			'value'         => '1',
			'no_found_rows' => true,
			'number'        => 9999999,
		);

		return get_posts( $args );
	}

	/**
	 * Generate the license keys for a payment during an ajax post
	 *
	 * @param \EDD\Orders\Order $order       The order object.
	 * @param int               $download_id The download ID.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	private function generate_license_keys( $order, $download_id = false ) {
		if ( ! $order ) {
			return esc_html__( 'No order found.', 'edd_sl' );
		}

		if ( empty( $order->items ) ) {
			return esc_html__( 'No order items found.', 'edd_sl' );
		}

		$existing_licenses     = edd_software_licensing()->get_licenses_of_purchase( $order->id );
		$generated_keys        = array();
		$number_keys_generated = 0;

		foreach ( $order->items as $order_item ) {

			if ( ! empty( $download_id ) && (int) $order_item->product_id !== (int) $download_id ) {
				continue; // We've been told to only generate for a specific download, and this wasn't it
			}
			$existing_license_count = 0;
			/**
			 * Count the number of licenses already generated for this payment/download/cart index combination.
			 *
			 * @since 3.8.6 (when the filter was added)
			 */
			if ( $existing_licenses && apply_filters( 'edd_sl_check_existing_licenses', true ) ) {
				foreach ( $existing_licenses as $existing_license ) {
					if ( $existing_license->cart_index !== $order_item->cart_index || $existing_license->download_id !== $order_item->product_id ) {
						continue;
					}
					++$existing_license_count;
				}
			}

			$item    = edd_get_download( $order_item->product_id );
			$type    = $item->is_bundled_download() ? 'bundle' : 'default';
			$license = edd_software_licensing()->get_license_by_purchase( $order->id, $order_item->product_id, $order_item->cart_index, false );

			if ( $license && $existing_license_count >= $order_item->quantity ) {

				if ( 'bundle' !== $type ) {
					continue;
				}

				$price_id = is_numeric( $license->price_id ) ? $license->price_id : null;

				// Pass in the expiration date so generated licenses have the correct expiration date.
				$options = array(
					'expiration_date' => $license->expiration,
				);

				// Always generate bundle license keys from the initial payment ID.
				$generated_keys = $license->create( $order_item->product_id, $license->order_id, $price_id, $license->cart_index, $options );
			} else {

				$args = array(
					'download_id' => $order_item->product_id,
					'payment_id'  => $order->id,
					'type'        => $type,
					'cart_item'   => $order_item,
					'cart_index'  => $order_item->cart_index,
					'retroactive' => true,
				);

				for ( $i = 0; $i <= $order_item->quantity; $i++ ) {
					$generated_keys = array_merge( $generated_keys, $this->generate_new_license( $args ) );
				}
			}

			$number_keys_generated = $number_keys_generated + count( $generated_keys );
		}

		$this->get_generated_keys( $generated_keys );

		return $generated_keys;
	}

	/**
	 * Get the generated keys.
	 *
	 * @since 3.9.0
	 * @param array $keys The newly generated keys.
	 * @return array The generated keys.
	 */
	private function get_generated_keys( array $keys ) {
		$transient = 'edd_sl_generated_keys';
		if ( ! empty( $this->product ) ) {
			$transient .= '_' . $this->product;
		}
		$generated_keys = get_transient( $transient );
		if ( empty( $keys ) ) {
			return $generated_keys;
		}

		if ( ! $generated_keys ) {
			$generated_keys = array();
		}

		$generated_keys = array_merge( $generated_keys, $keys );

		// Store the generated keys in a transient so they can be used in the response.
		set_transient( $transient, $generated_keys, HOUR_IN_SECONDS );

		return $generated_keys;
	}

	/**
	 * Delete the transient.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	private function delete_transient() {
		$transient = 'edd_sl_generated_keys';
		if ( ! empty( $this->product ) ) {
			$transient .= '_' . $this->product;
		}

		delete_transient( $transient );
	}
}
